"use strict";

/**
 * Job Application Tracker (Offline)
 * - Vanilla JS + LocalStorage
 * - Search + filter + stats
 * - Follow-up due badges
 * - Export/Import (JSON + CSV)
 * - Drag & drop (desktop) + status dropdown (mobile)
 *
 * FIXES INCLUDED:
 * 1) Modal close reliably (CSS can override [hidden], so we force display none/grid).
 * 2) White UI default theme (light).
 * 3) Demo/sample data auto-seeds only once when storage is empty (good for screenshots).
 * 4) Clear button wipes data + demo-seeded flag so users can start fresh.
 */

const STORAGE_KEY = "jat_v1_apps";
const THEME_KEY = "jat_v1_theme";
const DEMO_SEED_KEY = "jat_v1_seeded_demo";

const STATUSES = [
  { key: "applied",  label: "Applied" },
  { key: "interview", label: "Interview" },
  { key: "offer",    label: "Offer" },
  { key: "rejected", label: "Rejected" },
  { key: "archived", label: "Archived" },
];

const DEFAULT_FOLLOWUP_DAYS = 7;

// --------- DOM ----------
const boardEl = document.getElementById("board");
const qEl = document.getElementById("q");
const statusFilterEl = document.getElementById("statusFilter");
const sortByEl = document.getElementById("sortBy");
const dueOnlyEl = document.getElementById("dueOnly");

const statTotalEl = document.getElementById("statTotal");
const statAppliedEl = document.getElementById("statApplied");
const statInterviewEl = document.getElementById("statInterview");
const statOfferEl = document.getElementById("statOffer");
const statRejectedEl = document.getElementById("statRejected");
const statDueEl = document.getElementById("statDue");

const themeToggleEl = document.getElementById("themeToggle");

const openAddEl = document.getElementById("openAdd");
const modalBackdropEl = document.getElementById("modalBackdrop");
const closeModalEl = document.getElementById("closeModal");
const cancelBtnEl = document.getElementById("cancelBtn");
const formEl = document.getElementById("appForm");

const appIdEl = document.getElementById("appId");
const companyEl = document.getElementById("company");
const roleEl = document.getElementById("role");
const statusEl = document.getElementById("status");
const appliedDateEl = document.getElementById("appliedDate");
const linkEl = document.getElementById("link");
const followUpDaysEl = document.getElementById("followUpDays");
const contactEl = document.getElementById("contact");
const notesEl = document.getElementById("notes");

const exportJsonEl = document.getElementById("exportJson");
const exportCsvEl = document.getElementById("exportCsv");
const importFileEl = document.getElementById("importFile");
const clearAllEl = document.getElementById("clearAll");

// --------- State ----------
let apps = loadApps();
let dragId = null;

// --------- Init ----------
initTheme();
hydrateSelects();
wireEvents();

// Make sure modal starts hidden visually (even if CSS overrides [hidden])
setModalOpen(false);

// Seed demo data (only if empty + not seeded before)
maybeSeedDemoData();

render();

// --------- Helpers ----------
function nowISO(){
  return new Date().toISOString();
}

function todayYMD(){
  const d = new Date();
  const yyyy = d.getFullYear();
  const mm = String(d.getMonth()+1).padStart(2,"0");
  const dd = String(d.getDate()).padStart(2,"0");
  return `${yyyy}-${mm}-${dd}`;
}

function ymdDaysAgo(daysAgo){
  const d = new Date();
  d.setDate(d.getDate() - daysAgo);
  const yyyy = d.getFullYear();
  const mm = String(d.getMonth()+1).padStart(2,"0");
  const dd = String(d.getDate()).padStart(2,"0");
  return `${yyyy}-${mm}-${dd}`;
}

function parseYMD(ymd){
  if (!ymd) return null;
  const [y,m,d] = ymd.split("-").map(Number);
  if (!y || !m || !d) return null;
  return new Date(y, m-1, d);
}

function daysBetween(a, b){
  const ms = 24 * 60 * 60 * 1000;
  const da = new Date(a.getFullYear(), a.getMonth(), a.getDate());
  const db = new Date(b.getFullYear(), b.getMonth(), b.getDate());
  return Math.floor((db - da) / ms);
}

function addDays(date, days){
  const d = new Date(date);
  d.setDate(d.getDate() + days);
  return d;
}

function safeURL(url){
  try{
    if (!url) return "";
    const u = new URL(url);
    return u.toString();
  }catch(_){
    return "";
  }
}

function uid(){
  if (crypto && crypto.randomUUID) return crypto.randomUUID();
  return "id_" + Math.random().toString(16).slice(2) + "_" + Date.now().toString(16);
}

function loadApps(){
  try{
    const raw = localStorage.getItem(STORAGE_KEY);
    if (!raw) return [];
    const parsed = JSON.parse(raw);
    if (!Array.isArray(parsed)) return [];
    return parsed.filter(Boolean);
  }catch(_){
    return [];
  }
}

function saveApps(){
  localStorage.setItem(STORAGE_KEY, JSON.stringify(apps));
}

function isFollowUpDue(app){
  if (!app.followUpDays || app.followUpDays <= 0) return false;
  if (app.status === "rejected" || app.status === "archived" || app.status === "offer") return false;

  const applied = parseYMD(app.appliedDate) || parseYMD(todayYMD());
  const dueDate = addDays(applied, Number(app.followUpDays));
  const today = parseYMD(todayYMD());
  return today >= dueDate;
}

function followUpDueText(app){
  const applied = parseYMD(app.appliedDate);
  if (!applied || !app.followUpDays || app.followUpDays <= 0) return "";
  const due = addDays(applied, Number(app.followUpDays));
  const today = parseYMD(todayYMD());
  const diff = daysBetween(due, today);
  if (diff < 0) return `Follow-up in ${Math.abs(diff)}d`;
  if (diff === 0) return "Follow-up today";
  return `Follow-up overdue ${diff}d`;
}

function normalizeForSearch(s){
  return String(s || "").toLowerCase().trim();
}

function filteredApps(){
  const q = normalizeForSearch(qEl.value);
  const st = statusFilterEl.value;
  const dueOnly = dueOnlyEl.checked;

  let out = [...apps];

  if (st && st !== "all"){
    out = out.filter(a => a.status === st);
  }

  if (q){
    out = out.filter(a => {
      const hay = [a.company, a.role, a.notes, a.contact, a.link]
        .map(normalizeForSearch).join(" ");
      return hay.includes(q);
    });
  }

  if (dueOnly){
    out = out.filter(isFollowUpDue);
  }

  const sortBy = sortByEl.value;
  out.sort((a,b) => {
    if (sortBy === "updated_desc"){
      return String(b.updatedAt||"").localeCompare(String(a.updatedAt||""));
    }
    if (sortBy === "applied_desc"){
      return String(b.appliedDate||"").localeCompare(String(a.appliedDate||""));
    }
    if (sortBy === "applied_asc"){
      return String(a.appliedDate||"").localeCompare(String(b.appliedDate||""));
    }
    if (sortBy === "company_asc"){
      return String(a.company||"").localeCompare(String(b.company||""));
    }
    return 0;
  });

  return out;
}

function countsByStatus(items){
  const map = Object.fromEntries(STATUSES.map(s => [s.key, 0]));
  for (const it of items){
    if (map[it.status] !== undefined) map[it.status] += 1;
  }
  return map;
}

function updateStats(){
  const all = apps;
  const counts = countsByStatus(all);
  const dueCount = all.filter(isFollowUpDue).length;

  statTotalEl.textContent = String(all.length);
  statAppliedEl.textContent = String(counts.applied || 0);
  statInterviewEl.textContent = String(counts.interview || 0);
  statOfferEl.textContent = String(counts.offer || 0);
  statRejectedEl.textContent = String(counts.rejected || 0);
  statDueEl.textContent = String(dueCount);
}

// --------- Demo Seed ----------
function maybeSeedDemoData(){
  // Only seed if empty and we haven't seeded before
  if (apps.length > 0) return;
  if (localStorage.getItem(DEMO_SEED_KEY) === "1") return;

  apps = buildDemoData();
  saveApps();
  localStorage.setItem(DEMO_SEED_KEY, "1");
}

function buildDemoData(){
  // Real-looking sample entries (generic + safe)
  const base = nowISO();

  const demo = [
    {
      id: uid(),
      company: "Northbridge Analytics",
      role: "Frontend Developer (Next.js)",
      status: "applied",
      appliedDate: ymdDaysAgo(3),
      link: "",
      followUpDays: 7,
      contact: "Talent Team",
      notes: "Tailored CV for product UI work. Emphasized dashboards + accessibility.",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "Aster Health Systems",
      role: "Web Platform Coordinator",
      status: "interview",
      appliedDate: ymdDaysAgo(10),
      link: "",
      followUpDays: 5,
      contact: "M. Okoye (Recruiter)",
      notes: "Interview scheduled. Prep: SDLC, stakeholder updates, risk register examples.",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "KitePay Fintech",
      role: "Full-Stack Developer (PHP/Laravel)",
      status: "applied",
      appliedDate: ymdDaysAgo(6),
      link: "",
      followUpDays: 7,
      contact: "",
      notes: "Need to highlight payments integrations + secure auth patterns.",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "Evergreen Logistics",
      role: "Web Developer (Admin Portal)",
      status: "rejected",
      appliedDate: ymdDaysAgo(18),
      link: "",
      followUpDays: 0,
      contact: "",
      notes: "Rejection received. Keep in pipeline for future openings.",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "BrightPath NGO",
      role: "Website & Digital Delivery Volunteer",
      status: "interview",
      appliedDate: ymdDaysAgo(8),
      link: "",
      followUpDays: 4,
      contact: "Programme Lead",
      notes: "Discuss scope: landing refresh, form spam protection, performance checklist.",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "Helio Commerce",
      role: "UI Engineer (Vanilla JS)",
      status: "applied",
      appliedDate: ymdDaysAgo(2),
      link: "",
      followUpDays: 6,
      contact: "",
      notes: "Shared GitHub: kanban + calculators + this tracker.",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "Stonegate Studios",
      role: "Junior Product Coordinator",
      status: "archived",
      appliedDate: ymdDaysAgo(25),
      link: "",
      followUpDays: 0,
      contact: "",
      notes: "Archived (role paused).",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "CedarWorks GmbH",
      role: "Web Developer (WordPress)",
      status: "offer",
      appliedDate: ymdDaysAgo(14),
      link: "",
      followUpDays: 0,
      contact: "HR Desk",
      notes: "Offer received. Reviewing timeline + responsibilities and asking for details.",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "Orbit Media Lab",
      role: "Frontend Intern",
      status: "rejected",
      appliedDate: ymdDaysAgo(12),
      link: "",
      followUpDays: 0,
      contact: "",
      notes: "No match this round. Follow company for future roles.",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "SummitCare Partners",
      role: "Digital Projects Assistant",
      status: "interview",
      appliedDate: ymdDaysAgo(5),
      link: "",
      followUpDays: 3,
      contact: "Operations",
      notes: "Prep: reporting cadence, meeting minutes structure, rollout checklist.",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "BlueRiver Software",
      role: "Backend Developer (Flask)",
      status: "applied",
      appliedDate: ymdDaysAgo(1),
      link: "",
      followUpDays: 7,
      contact: "",
      notes: "Mentioned API design + billing routes + clean error handling.",
      createdAt: base, updatedAt: base
    },
    {
      id: uid(),
      company: "MapleWave Tech",
      role: "Junior Web Developer",
      status: "rejected",
      appliedDate: ymdDaysAgo(20),
      link: "",
      followUpDays: 0,
      contact: "",
      notes: "Rejected. Update portfolio screenshots for stronger first impression.",
      createdAt: base, updatedAt: base
    },
  ];

  // Put newest first (like a real tracker)
  demo.sort((a,b) => String(b.appliedDate).localeCompare(String(a.appliedDate)));
  return demo;
}

// --------- Render ----------
function render(){
  updateStats();

  const items = filteredApps();

  const grouped = Object.fromEntries(STATUSES.map(s => [s.key, []]));
  for (const app of items){
    (grouped[app.status] || grouped.applied).push(app);
  }

  boardEl.innerHTML = "";

  for (const st of STATUSES){
    const col = document.createElement("div");
    col.className = "column";
    col.dataset.status = st.key;

    const head = document.createElement("div");
    head.className = "column-head";

    const title = document.createElement("div");
    title.className = "column-title";
    title.textContent = st.label;

    const count = document.createElement("div");
    count.className = "count";
    count.textContent = String((grouped[st.key] || []).length);

    head.appendChild(title);
    head.appendChild(count);

    const zone = document.createElement("div");
    zone.className = "dropzone";
    zone.dataset.status = st.key;

    col.addEventListener("dragover", (e) => {
      e.preventDefault();
      col.classList.add("dragover");
    });
    col.addEventListener("dragleave", () => col.classList.remove("dragover"));
    col.addEventListener("drop", (e) => {
      e.preventDefault();
      col.classList.remove("dragover");
      if (!dragId) return;
      moveToStatus(dragId, st.key);
      dragId = null;
    });

    for (const app of grouped[st.key] || []){
      zone.appendChild(renderCard(app));
    }

    col.appendChild(head);
    col.appendChild(zone);
    boardEl.appendChild(col);
  }
}

function renderCard(app){
  const card = document.createElement("article");
  card.className = "card";
  card.draggable = true;
  card.dataset.id = app.id;
  card.dataset.status = app.status; // for status stripe color in CSS

  card.addEventListener("dragstart", () => {
    dragId = app.id;
  });

  const top = document.createElement("div");
  top.className = "card-top";

  const left = document.createElement("div");

  const h = document.createElement("h3");
  h.className = "card-title";
  h.textContent = app.company || "(No company)";

  const sub = document.createElement("p");
  sub.className = "card-sub";
  sub.textContent = app.role || "(No role)";

  left.appendChild(h);
  left.appendChild(sub);

  const actions = document.createElement("div");
  actions.className = "card-actions";

  const openBtn = document.createElement("button");
  openBtn.className = "icon-btn";
  openBtn.type = "button";
  openBtn.title = "Open link";
  openBtn.textContent = "↗";
  openBtn.disabled = !safeURL(app.link);
  openBtn.addEventListener("click", () => {
    const u = safeURL(app.link);
    if (u) window.open(u, "_blank", "noopener,noreferrer");
  });

  const editBtn = document.createElement("button");
  editBtn.className = "icon-btn";
  editBtn.type = "button";
  editBtn.title = "Edit";
  editBtn.textContent = "✎";
  editBtn.addEventListener("click", () => openModalForEdit(app.id));

  const delBtn = document.createElement("button");
  delBtn.className = "icon-btn";
  delBtn.type = "button";
  delBtn.title = "Delete";
  delBtn.textContent = "🗑";
  delBtn.addEventListener("click", () => {
    const ok = confirm(`Delete "${app.company}" – "${app.role}"?`);
    if (!ok) return;
    apps = apps.filter(a => a.id !== app.id);
    saveApps();
    render();
  });

  actions.appendChild(openBtn);
  actions.appendChild(editBtn);
  actions.appendChild(delBtn);

  top.appendChild(left);
  top.appendChild(actions);

  const badges = document.createElement("div");
  badges.className = "badges";

  const applied = parseYMD(app.appliedDate);
  if (applied){
    const diff = daysBetween(applied, parseYMD(todayYMD()));
    const b = document.createElement("span");
    b.className = "badge";
    b.textContent = `Applied ${diff}d ago`;
    badges.appendChild(b);
  }

  if (app.contact){
    const b = document.createElement("span");
    b.className = "badge";
    b.textContent = app.contact;
    badges.appendChild(b);
  }

  if (app.followUpDays && app.followUpDays > 0){
    const b = document.createElement("span");
    const due = isFollowUpDue(app);
    b.className = "badge" + (due ? " due" : "");
    b.textContent = followUpDueText(app);
    badges.appendChild(b);
  }

  const bottom = document.createElement("div");
  bottom.className = "card-bottom";

  const small = document.createElement("div");
  small.className = "small";
  small.textContent = app.appliedDate ? `Applied: ${app.appliedDate}` : "Applied: —";

  const statusSel = document.createElement("select");
  for (const st of STATUSES){
    const opt = document.createElement("option");
    opt.value = st.key;
    opt.textContent = st.label;
    statusSel.appendChild(opt);
  }
  statusSel.value = app.status;
  statusSel.title = "Change status";
  statusSel.addEventListener("change", () => {
    moveToStatus(app.id, statusSel.value);
  });

  bottom.appendChild(small);
  bottom.appendChild(statusSel);

  card.appendChild(top);
  card.appendChild(badges);

  if (app.notes){
    const p = document.createElement("p");
    p.className = "card-sub";
    p.style.marginTop = "10px";
    p.textContent = truncate(app.notes, 140);
    card.appendChild(p);
  }

  card.appendChild(bottom);
  return card;
}

function truncate(s, n){
  const str = String(s || "").trim();
  if (str.length <= n) return str;
  return str.slice(0, n-1) + "…";
}

// --------- CRUD ----------
function openModalForAdd(){
  document.getElementById("modalTitle").textContent = "Add application";
  appIdEl.value = "";
  companyEl.value = "";
  roleEl.value = "";
  statusEl.value = "applied";
  appliedDateEl.value = todayYMD();
  linkEl.value = "";
  followUpDaysEl.value = String(DEFAULT_FOLLOWUP_DAYS);
  contactEl.value = "";
  notesEl.value = "";
  openModal();
}

function openModalForEdit(id){
  const app = apps.find(a => a.id === id);
  if (!app) return;

  document.getElementById("modalTitle").textContent = "Edit application";
  appIdEl.value = app.id;
  companyEl.value = app.company || "";
  roleEl.value = app.role || "";
  statusEl.value = app.status || "applied";
  appliedDateEl.value = app.appliedDate || "";
  linkEl.value = app.link || "";
  followUpDaysEl.value = (app.followUpDays ?? "").toString();
  contactEl.value = app.contact || "";
  notesEl.value = app.notes || "";
  openModal();
}

function upsertFromForm(){
  const id = appIdEl.value || uid();
  const company = companyEl.value.trim();
  const role = roleEl.value.trim();
  const status = statusEl.value || "applied";
  const appliedDate = appliedDateEl.value || "";
  const link = linkEl.value.trim();
  const followUpDaysRaw = followUpDaysEl.value.trim();
  const followUpDays = followUpDaysRaw === "" ? 0 : Math.max(0, Math.min(365, Number(followUpDaysRaw)));
  const contact = contactEl.value.trim();
  const notes = notesEl.value.trim();

  if (!company || !role){
    alert("Please fill Company and Role.");
    return null;
  }

  const cleanLink = link ? (safeURL(link) || "") : "";

  const existingIndex = apps.findIndex(a => a.id === id);
  const base = existingIndex >= 0 ? apps[existingIndex] : null;

  const record = {
    id,
    company,
    role,
    status,
    appliedDate,
    link: cleanLink,
    followUpDays,
    contact,
    notes,
    createdAt: base?.createdAt || nowISO(),
    updatedAt: nowISO(),
  };

  if (existingIndex >= 0) apps[existingIndex] = record;
  else apps.unshift(record);

  saveApps();
  render();
  return record;
}

function moveToStatus(id, status){
  const i = apps.findIndex(a => a.id === id);
  if (i < 0) return;
  apps[i] = { ...apps[i], status, updatedAt: nowISO() };
  saveApps();
  render();
}

// --------- Import/Export ----------
function download(filename, content, mime){
  const blob = new Blob([content], { type: mime });
  const url = URL.createObjectURL(blob);
  const a = document.createElement("a");
  a.href = url;
  a.download = filename;
  document.body.appendChild(a);
  a.click();
  a.remove();
  URL.revokeObjectURL(url);
}

function exportJSON(){
  const payload = { version: 1, exportedAt: nowISO(), items: apps };
  download(`job-tracker-export-${todayYMD()}.json`, JSON.stringify(payload, null, 2), "application/json");
}

function csvEscape(v){
  const s = String(v ?? "");
  if (/[",\n]/.test(s)) return `"${s.replace(/"/g,'""')}"`;
  return s;
}

function exportCSV(){
  const headers = ["company","role","status","appliedDate","followUpDays","contact","link","notes","createdAt","updatedAt"];
  const rows = [headers.join(",")];

  for (const a of apps){
    rows.push(headers.map(h => csvEscape(a[h])).join(","));
  }

  download(`job-tracker-export-${todayYMD()}.csv`, rows.join("\n"), "text/csv");
}

async function importJSONFile(file){
  const text = await file.text();
  let parsed;
  try { parsed = JSON.parse(text); }
  catch { alert("Invalid JSON file."); return; }

  const incoming = Array.isArray(parsed) ? parsed : parsed.items;
  if (!Array.isArray(incoming)){
    alert("JSON format not recognized. Expected an array or { items: [...] }.");
    return;
  }

  const normalized = incoming
    .filter(Boolean)
    .map(it => ({
      id: it.id || uid(),
      company: String(it.company || "").trim(),
      role: String(it.role || "").trim(),
      status: STATUSES.some(s => s.key === it.status) ? it.status : "applied",
      appliedDate: String(it.appliedDate || "").trim(),
      link: safeURL(String(it.link || "").trim()) || "",
      followUpDays: Number(it.followUpDays || 0) || 0,
      contact: String(it.contact || "").trim(),
      notes: String(it.notes || "").trim(),
      createdAt: it.createdAt || nowISO(),
      updatedAt: it.updatedAt || nowISO(),
    }))
    .filter(it => it.company && it.role);

  if (!normalized.length){
    alert("No valid items found in import.");
    return;
  }

  const doReplace = confirm("Replace your current data with this import?\n\nOK = Replace\nCancel = Merge");
  if (doReplace){
    apps = normalized;
  }else{
    const map = new Map(apps.map(a => [a.id, a]));
    for (const n of normalized) map.set(n.id, n);
    apps = Array.from(map.values()).sort((a,b) => String(b.updatedAt||"").localeCompare(String(a.updatedAt||"")));
  }

  saveApps();
  render();
  alert(`Imported ${normalized.length} item(s).`);
}

// --------- Theme ----------
function initTheme(){
  const saved = localStorage.getItem(THEME_KEY);
  const theme = saved || "light"; // white UI default
  document.documentElement.setAttribute("data-theme", theme);
}

function toggleTheme(){
  const current = document.documentElement.getAttribute("data-theme") || "light";
  const next = current === "dark" ? "light" : "dark";
  document.documentElement.setAttribute("data-theme", next);
  localStorage.setItem(THEME_KEY, next);
}

// --------- Modal (FIXED) ----------
function setModalOpen(isOpen){
  if (!modalBackdropEl) return;

  if (isOpen){
    modalBackdropEl.hidden = false;
    modalBackdropEl.style.display = "grid";
    document.body.style.overflow = "hidden";
  }else{
    modalBackdropEl.hidden = true;
    modalBackdropEl.style.display = "none";
    document.body.style.overflow = "";
  }
}

function openModal(){
  setModalOpen(true);
  setTimeout(() => companyEl?.focus(), 0);
}

function closeModal(){
  setModalOpen(false);
}

// --------- UI wiring ----------
function hydrateSelects(){
  statusEl.innerHTML = "";
  for (const st of STATUSES){
    const opt = document.createElement("option");
    opt.value = st.key;
    opt.textContent = st.label;
    statusEl.appendChild(opt);
  }
  statusEl.value = "applied";

  for (const st of STATUSES){
    const opt = document.createElement("option");
    opt.value = st.key;
    opt.textContent = st.label;
    statusFilterEl.appendChild(opt);
  }
}

function wireEvents(){
  qEl.addEventListener("input", render);
  statusFilterEl.addEventListener("change", render);
  sortByEl.addEventListener("change", render);
  dueOnlyEl.addEventListener("change", render);

  themeToggleEl.addEventListener("click", toggleTheme);

  openAddEl.addEventListener("click", openModalForAdd);

  closeModalEl.addEventListener("click", (e) => { e.preventDefault(); closeModal(); });
  cancelBtnEl.addEventListener("click", (e) => { e.preventDefault(); closeModal(); });

  modalBackdropEl.addEventListener("click", (e) => {
    if (e.target === modalBackdropEl) closeModal();
  });

  document.addEventListener("keydown", (e) => {
    if (e.key === "Escape" && modalBackdropEl.style.display !== "none") closeModal();
  });

  formEl.addEventListener("submit", (e) => {
    e.preventDefault();
    const rec = upsertFromForm();
    if (rec) closeModal();
  });

  exportJsonEl.addEventListener("click", exportJSON);
  exportCsvEl.addEventListener("click", exportCSV);

  importFileEl.addEventListener("change", async (e) => {
    const file = e.target.files && e.target.files[0];
    if (!file) return;
    await importJSONFile(file);
    importFileEl.value = "";
  });

  clearAllEl.addEventListener("click", () => {
    const ok = confirm("Clear all saved applications from this browser?\n\nTip: Export first if you want a backup.");
    if (!ok) return;

    apps = [];
    localStorage.removeItem(STORAGE_KEY);
    localStorage.removeItem(DEMO_SEED_KEY); // allow demo to show again on next refresh (optional)
    render();
  });
}
