<?php

declare(strict_types=1);

namespace App\Auth;

use App\Database\DB;
use App\Session\SessionManager;

class Auth
{
    private const BCRYPT_COST = 12;
    private const MAX_FAILED_ATTEMPTS = 5;
    private const LOCKOUT_DURATION = 1800;

    public static function register(string $name, string $email, string $password): array
    {
        $existing = DB::fetch('SELECT id FROM users WHERE email = ?', [$email]);
        if ($existing) {
            return ['ok' => false, 'error' => 'An account with this email already exists.'];
        }

        $hash = password_hash($password, PASSWORD_BCRYPT, ['cost' => self::BCRYPT_COST]);

        $id = DB::insert('users', [
            'name'       => $name,
            'email'      => strtolower(trim($email)),
            'password'   => $hash,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
        ]);

        AuditLog::log(AuditLog::REGISTER, (int) $id, ['email' => $email]);

        return ['ok' => true, 'user_id' => (int) $id];
    }

    public static function attempt(string $email, string $password): array
    {
        $user = DB::fetch('SELECT * FROM users WHERE email = ?', [strtolower(trim($email))]);

        if (!$user) {
            AuditLog::log(AuditLog::LOGIN_FAILED, null, ['email' => $email, 'reason' => 'no_account']);
            return ['ok' => false, 'error' => 'Invalid email or password.'];
        }

        if (self::isLocked($user)) {
            return ['ok' => false, 'error' => 'Account temporarily locked. Try again later.'];
        }

        if (!password_verify($password, $user['password'])) {
            self::incrementFailedAttempts($user['id']);
            AuditLog::log(AuditLog::LOGIN_FAILED, $user['id'], ['reason' => 'wrong_password']);
            return ['ok' => false, 'error' => 'Invalid email or password.'];
        }

        if ($user['two_factor_secret']) {
            SessionManager::set('2fa_user_id', $user['id']);
            return ['ok' => true, 'requires_2fa' => true];
        }

        self::loginUser($user);
        return ['ok' => true, 'requires_2fa' => false];
    }

    public static function loginUser(array $user): void
    {
        SessionManager::regenerate();
        SessionManager::set('user_id', $user['id']);
        SessionManager::set('user_name', $user['name']);
        SessionManager::set('user_email', $user['email']);
        SessionManager::setFingerprint();

        self::resetFailedAttempts($user['id']);

        DB::update('users', ['last_login_at' => date('Y-m-d H:i:s')], 'id = ?', [$user['id']]);
        AuditLog::log(AuditLog::LOGIN_SUCCESS, $user['id']);
    }

    public static function logout(): void
    {
        $userId = SessionManager::get('user_id');
        if ($userId) {
            AuditLog::log(AuditLog::LOGOUT, $userId);
        }

        DB::query('DELETE FROM remember_tokens WHERE user_id = ?', [$userId]);
        SessionManager::destroy();
    }

    public static function user(): ?array
    {
        $id = SessionManager::get('user_id');
        if (!$id) {
            return null;
        }
        return DB::fetch('SELECT id, name, email, two_factor_secret, created_at, last_login_at FROM users WHERE id = ?', [$id]);
    }

    public static function createRememberToken(int $userId): string
    {
        $token = bin2hex(random_bytes(32));
        $hash  = hash('sha256', $token);

        DB::query('DELETE FROM remember_tokens WHERE user_id = ?', [$userId]);

        DB::insert('remember_tokens', [
            'user_id'    => $userId,
            'token_hash' => $hash,
            'expires_at' => date('Y-m-d H:i:s', time() + (30 * 86400)),
            'created_at' => date('Y-m-d H:i:s'),
        ]);

        return $token;
    }

    public static function loginFromRememberToken(string $token): bool
    {
        $hash = hash('sha256', $token);

        $row = DB::fetch(
            'SELECT user_id FROM remember_tokens WHERE token_hash = ? AND expires_at > NOW()',
            [$hash]
        );

        if (!$row) {
            return false;
        }

        $user = DB::fetch('SELECT * FROM users WHERE id = ?', [$row['user_id']]);
        if (!$user) {
            return false;
        }

        DB::query('DELETE FROM remember_tokens WHERE token_hash = ?', [$hash]);
        self::loginUser($user);

        $newToken = self::createRememberToken($user['id']);
        setcookie('remember_token', $newToken, [
            'expires'  => time() + (30 * 86400),
            'path'     => '/',
            'httponly'  => true,
            'samesite' => 'Lax',
        ]);

        return true;
    }

    private static function isLocked(array $user): bool
    {
        if ($user['failed_attempts'] < self::MAX_FAILED_ATTEMPTS) {
            return false;
        }

        $lockedUntil = strtotime($user['locked_until'] ?? '');
        if ($lockedUntil && $lockedUntil > time()) {
            return true;
        }

        if ($lockedUntil && $lockedUntil <= time()) {
            self::resetFailedAttempts($user['id']);
            return false;
        }

        return false;
    }

    private static function incrementFailedAttempts(int $userId): void
    {
        DB::query('UPDATE users SET failed_attempts = failed_attempts + 1 WHERE id = ?', [$userId]);

        $user = DB::fetch('SELECT failed_attempts FROM users WHERE id = ?', [$userId]);

        if ($user && $user['failed_attempts'] >= self::MAX_FAILED_ATTEMPTS) {
            $lockUntil = date('Y-m-d H:i:s', time() + self::LOCKOUT_DURATION);
            DB::update('users', ['locked_until' => $lockUntil], 'id = ?', [$userId]);
            AuditLog::log(AuditLog::ACCOUNT_LOCKED, $userId);
        }
    }

    private static function resetFailedAttempts(int $userId): void
    {
        DB::update('users', [
            'failed_attempts' => 0,
            'locked_until'    => null,
        ], 'id = ?', [$userId]);
    }
}
