<?php

declare(strict_types=1);

namespace App\Auth;

use App\Database\DB;

class PasswordReset
{
    private const TOKEN_LIFETIME = 3600;

    public static function createToken(string $email): ?string
    {
        $user = DB::fetch('SELECT id FROM users WHERE email = ?', [strtolower(trim($email))]);
        if (!$user) {
            return null;
        }

        DB::query('DELETE FROM password_resets WHERE email = ?', [strtolower(trim($email))]);

        $token = bin2hex(random_bytes(32));
        $hash  = hash('sha256', $token);

        DB::insert('password_resets', [
            'email'      => strtolower(trim($email)),
            'token_hash' => $hash,
            'expires_at' => date('Y-m-d H:i:s', time() + self::TOKEN_LIFETIME),
            'created_at' => date('Y-m-d H:i:s'),
        ]);

        AuditLog::log(AuditLog::PASSWORD_RESET_REQUEST, $user['id']);

        return $token;
    }

    public static function validate(string $token, string $email): bool
    {
        $hash = hash('sha256', $token);

        $row = DB::fetch(
            'SELECT * FROM password_resets WHERE token_hash = ? AND email = ? AND expires_at > NOW()',
            [$hash, strtolower(trim($email))]
        );

        return $row !== null;
    }

    public static function reset(string $token, string $email, string $newPassword): bool
    {
        if (!self::validate($token, $email)) {
            return false;
        }

        $hash = password_hash($newPassword, PASSWORD_BCRYPT, ['cost' => 12]);
        $email = strtolower(trim($email));

        DB::query('UPDATE users SET password = ?, failed_attempts = 0, locked_until = NULL, updated_at = NOW() WHERE email = ?', [$hash, $email]);
        DB::query('DELETE FROM password_resets WHERE email = ?', [$email]);

        $user = DB::fetch('SELECT id FROM users WHERE email = ?', [$email]);
        if ($user) {
            AuditLog::log(AuditLog::PASSWORD_RESET, $user['id']);
        }

        return true;
    }

    public static function cleanup(): void
    {
        DB::query('DELETE FROM password_resets WHERE expires_at < NOW()');
    }
}
