<?php

declare(strict_types=1);

namespace App\Auth;

use App\Database\DB;
use PragmaRX\Google2FA\Google2FA;
use BaconQrCode\Renderer\ImageRenderer;
use BaconQrCode\Renderer\Image\SvgImageBackEnd;
use BaconQrCode\Renderer\RendererStyle\RendererStyle;
use BaconQrCode\Writer;

class TwoFactor
{
    private static ?Google2FA $g2fa = null;

    private static function engine(): Google2FA
    {
        if (self::$g2fa === null) {
            self::$g2fa = new Google2FA();
        }
        return self::$g2fa;
    }

    public static function generateSecret(): string
    {
        return self::engine()->generateSecretKey();
    }

    public static function qrCodeSvg(string $secret, string $email, string $appName): string
    {
        $url = self::engine()->getQRCodeUrl($appName, $email, $secret);

        $renderer = new ImageRenderer(
            new RendererStyle(200),
            new SvgImageBackEnd()
        );

        $writer = new Writer($renderer);
        return $writer->writeString($url);
    }

    public static function verify(string $secret, string $code): bool
    {
        return self::engine()->verifyKey($secret, $code);
    }

    public static function enable(int $userId, string $secret): void
    {
        $encrypted = self::encrypt($secret);
        DB::update('users', ['two_factor_secret' => $encrypted, 'updated_at' => date('Y-m-d H:i:s')], 'id = ?', [$userId]);
        AuditLog::log(AuditLog::TWO_FA_ENABLED, $userId);
    }

    public static function disable(int $userId): void
    {
        DB::update('users', ['two_factor_secret' => null, 'updated_at' => date('Y-m-d H:i:s')], 'id = ?', [$userId]);
        AuditLog::log(AuditLog::TWO_FA_DISABLED, $userId);
    }

    public static function getSecret(int $userId): ?string
    {
        $user = DB::fetch('SELECT two_factor_secret FROM users WHERE id = ?', [$userId]);
        if (!$user || !$user['two_factor_secret']) {
            return null;
        }
        return self::decrypt($user['two_factor_secret']);
    }

    public static function verifyForUser(int $userId, string $code): bool
    {
        $secret = self::getSecret($userId);
        if (!$secret) {
            return false;
        }

        $valid = self::verify($secret, $code);

        AuditLog::log(
            $valid ? AuditLog::TWO_FA_VERIFIED : AuditLog::TWO_FA_FAILED,
            $userId
        );

        return $valid;
    }

    private static function encrypt(string $data): string
    {
        $key    = hash('sha256', config('key'), true);
        $iv     = random_bytes(16);
        $cipher = openssl_encrypt($data, 'aes-256-cbc', $key, OPENSSL_RAW_DATA, $iv);
        return base64_encode($iv . $cipher);
    }

    private static function decrypt(string $data): string
    {
        $key  = hash('sha256', config('key'), true);
        $raw  = base64_decode($data);
        $iv   = substr($raw, 0, 16);
        $text = substr($raw, 16);
        return openssl_decrypt($text, 'aes-256-cbc', $key, OPENSSL_RAW_DATA, $iv);
    }
}
