<?php

declare(strict_types=1);

namespace App\Auth;

class Validator
{
    private array $errors = [];

    public function email(string $value, string $field = 'email'): self
    {
        if (empty($value)) {
            $this->errors[$field] = 'Email is required.';
        } elseif (!filter_var($value, FILTER_VALIDATE_EMAIL)) {
            $this->errors[$field] = 'Enter a valid email address.';
        }
        return $this;
    }

    public function required(string $value, string $field, string $label = null): self
    {
        if (empty(trim($value))) {
            $this->errors[$field] = ($label ?? ucfirst($field)) . ' is required.';
        }
        return $this;
    }

    public function minLength(string $value, int $min, string $field, string $label = null): self
    {
        if (strlen($value) < $min) {
            $this->errors[$field] = ($label ?? ucfirst($field)) . " must be at least {$min} characters.";
        }
        return $this;
    }

    public function match(string $a, string $b, string $field, string $message = 'Fields do not match.'): self
    {
        if ($a !== $b) {
            $this->errors[$field] = $message;
        }
        return $this;
    }

    public function passwordStrength(string $password, string $field = 'password'): self
    {
        if (strlen($password) < 8) {
            $this->errors[$field] = 'Password must be at least 8 characters.';
        } elseif (!preg_match('/[A-Z]/', $password) || !preg_match('/[0-9]/', $password)) {
            $this->errors[$field] = 'Password needs at least one uppercase letter and one number.';
        }
        return $this;
    }

    public function fails(): bool
    {
        return !empty($this->errors);
    }

    public function errors(): array
    {
        return $this->errors;
    }

    public function firstError(): ?string
    {
        return !empty($this->errors) ? reset($this->errors) : null;
    }
}
