<?php

declare(strict_types=1);

namespace App\Mail;

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\SMTP;
use PHPMailer\PHPMailer\Exception;

class Mailer
{
    private array $config;

    public function __construct(array $smtpConfig)
    {
        $this->config = $smtpConfig;
    }

    public function send(string $to, string $subject, string $htmlBody, string $textBody = ''): bool
    {
        $mail = new PHPMailer(true);

        try {
            $mail->isSMTP();
            $mail->Host       = $this->config['host'];
            $mail->SMTPAuth   = true;
            $mail->Username   = $this->config['user'];
            $mail->Password   = $this->config['pass'];
            $mail->SMTPSecure = $this->config['encryption'] === 'tls' ? PHPMailer::ENCRYPTION_STARTTLS : PHPMailer::ENCRYPTION_SMTPS;
            $mail->Port       = $this->config['port'];

            $mail->setFrom($this->config['from_email'], $this->config['from_name']);
            $mail->addAddress($to);

            $mail->isHTML(true);
            $mail->Subject = $subject;
            $mail->Body    = $htmlBody;
            $mail->AltBody = $textBody ?: strip_tags($htmlBody);

            $mail->send();
            return true;
        } catch (Exception $e) {
            error_log("Mail send failed: {$mail->ErrorInfo}");
            return false;
        }
    }

    public function sendPasswordReset(string $to, string $resetUrl, string $appName): bool
    {
        $subject = "Reset your {$appName} password";

        $html = $this->wrapInLayout($appName, "
            <h2 style='margin:0 0 16px;color:#1a1a1a;font-size:22px;'>Password Reset</h2>
            <p style='margin:0 0 12px;color:#555;font-size:15px;line-height:1.6;'>
                Someone requested a password reset for your account. If this was you, click the button below.
                This link expires in 1 hour.
            </p>
            <a href='{$resetUrl}'
               style='display:inline-block;padding:12px 32px;background:#111;color:#fff;text-decoration:none;border-radius:8px;font-weight:600;font-size:14px;margin:16px 0;'>
                Reset Password
            </a>
            <p style='margin:16px 0 0;color:#999;font-size:13px;'>
                If you didn't request this, just ignore this email. Your password won't change.
            </p>
        ");

        return $this->send($to, $subject, $html);
    }

    public function sendWelcome(string $to, string $name, string $loginUrl, string $appName): bool
    {
        $subject = "Welcome to {$appName}";

        $html = $this->wrapInLayout($appName, "
            <h2 style='margin:0 0 16px;color:#1a1a1a;font-size:22px;'>Welcome, {$name}!</h2>
            <p style='margin:0 0 12px;color:#555;font-size:15px;line-height:1.6;'>
                Your account has been created. You can now log in and get started.
            </p>
            <a href='{$loginUrl}'
               style='display:inline-block;padding:12px 32px;background:#111;color:#fff;text-decoration:none;border-radius:8px;font-weight:600;font-size:14px;margin:16px 0;'>
                Sign In
            </a>
        ");

        return $this->send($to, $subject, $html);
    }

    private function wrapInLayout(string $appName, string $content): string
    {
        return "
        <!DOCTYPE html>
        <html>
        <head><meta charset='utf-8'></head>
        <body style='margin:0;padding:0;background:#f4f4f5;font-family:-apple-system,BlinkMacSystemFont,Segoe UI,Roboto,sans-serif;'>
            <div style='max-width:520px;margin:40px auto;background:#fff;border-radius:12px;padding:40px;box-shadow:0 1px 3px rgba(0,0,0,0.08);'>
                <div style='margin-bottom:24px;font-weight:700;font-size:18px;color:#111;'>{$appName}</div>
                {$content}
                <hr style='border:none;border-top:1px solid #eee;margin:32px 0 16px;'>
                <p style='margin:0;color:#aaa;font-size:12px;'>
                    This email was sent by {$appName}. If you have questions, reply to this email.
                </p>
            </div>
        </body>
        </html>";
    }
}
