<?php

declare(strict_types=1);

namespace App\Middleware;

use App\Database\DB;

class RateLimiter
{
    public static function hit(string $action, string $identifier = null): void
    {
        $identifier = $identifier ?? self::ip();

        DB::insert('rate_limits', [
            'action'     => $action,
            'identifier' => $identifier,
            'created_at' => date('Y-m-d H:i:s'),
        ]);
    }

    public static function tooMany(string $action, int $maxAttempts, int $windowSeconds, string $identifier = null): bool
    {
        $identifier = $identifier ?? self::ip();
        $since = date('Y-m-d H:i:s', time() - $windowSeconds);

        $row = DB::fetch(
            'SELECT COUNT(*) as attempts FROM rate_limits WHERE action = ? AND identifier = ? AND created_at > ?',
            [$action, $identifier, $since]
        );

        return ($row['attempts'] ?? 0) >= $maxAttempts;
    }

    public static function clear(string $action, string $identifier = null): void
    {
        $identifier = $identifier ?? self::ip();
        DB::query('DELETE FROM rate_limits WHERE action = ? AND identifier = ?', [$action, $identifier]);
    }

    public static function cleanup(): void
    {
        $cutoff = date('Y-m-d H:i:s', time() - 86400);
        DB::query('DELETE FROM rate_limits WHERE created_at < ?', [$cutoff]);
    }

    public static function check(string $action, int $maxAttempts, int $windowSeconds): void
    {
        if (self::tooMany($action, $maxAttempts, $windowSeconds)) {
            http_response_code(429);
            $minutes = ceil($windowSeconds / 60);
            die("Too many attempts. Please wait {$minutes} minutes before trying again.");
        }
    }

    private static function ip(): string
    {
        return $_SERVER['HTTP_X_FORWARDED_FOR'] ?? $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0';
    }
}
