<?php

declare(strict_types=1);

namespace App\Session;

class SessionManager
{
    public static function start(array $config): void
    {
        if (session_status() === PHP_SESSION_ACTIVE) {
            return;
        }

        $lifetime = $config['lifetime'] * 60;

        ini_set('session.gc_maxlifetime', (string) $lifetime);
        ini_set('session.cookie_httponly', '1');
        ini_set('session.cookie_samesite', 'Lax');
        ini_set('session.use_strict_mode', '1');
        ini_set('session.use_only_cookies', '1');

        if ($config['secure']) {
            ini_set('session.cookie_secure', '1');
        }

        session_save_path(dirname(__DIR__, 2) . '/storage/sessions');
        session_start();

        if (self::isExpired($lifetime)) {
            self::destroy();
            session_start();
        }

        $_SESSION['_last_activity'] = time();
    }

    public static function regenerate(): void
    {
        session_regenerate_id(true);
    }

    public static function set(string $key, mixed $value): void
    {
        $_SESSION[$key] = $value;
    }

    public static function get(string $key, mixed $default = null): mixed
    {
        return $_SESSION[$key] ?? $default;
    }

    public static function has(string $key): bool
    {
        return isset($_SESSION[$key]);
    }

    public static function remove(string $key): void
    {
        unset($_SESSION[$key]);
    }

    public static function flash(string $key, mixed $value): void
    {
        $_SESSION['_flash'][$key] = $value;
    }

    public static function getFlash(string $key, mixed $default = null): mixed
    {
        $value = $_SESSION['_flash'][$key] ?? $default;
        unset($_SESSION['_flash'][$key]);
        return $value;
    }

    public static function destroy(): void
    {
        $_SESSION = [];

        if (ini_get('session.use_cookies')) {
            $params = session_get_cookie_params();
            setcookie(
                session_name(),
                '',
                time() - 42000,
                $params['path'],
                $params['domain'],
                $params['secure'],
                $params['httponly']
            );
        }

        session_destroy();
    }

    public static function fingerprint(): string
    {
        return hash('sha256', ($_SERVER['HTTP_USER_AGENT'] ?? '') . ($_SERVER['HTTP_ACCEPT_LANGUAGE'] ?? ''));
    }

    public static function validateFingerprint(): bool
    {
        $stored = self::get('_fingerprint');
        if ($stored === null) {
            return true;
        }
        return hash_equals($stored, self::fingerprint());
    }

    public static function setFingerprint(): void
    {
        self::set('_fingerprint', self::fingerprint());
    }

    private static function isExpired(int $lifetime): bool
    {
        $last = $_SESSION['_last_activity'] ?? null;
        if ($last === null) {
            return false;
        }
        return (time() - $last) > $lifetime;
    }
}
