# API Documentation

## Overview

The Scientific Calculator & Unit Converter provides PHP classes that can be easily integrated into other projects.

## Installation

### Using in Your Project

1. Copy the `src` directory to your project:
```bash
cp -r src/ /path/to/your/project/
```

2. Require the classes in your PHP file:
```php
require_once 'src/Calculator.php';
require_once 'src/UnitConverter.php';

use ScientificCalc\Calculator;
use ScientificCalc\UnitConverter;
```

## Calculator Class

### Namespace
```php
ScientificCalc\Calculator
```

### Constructor
```php
$calculator = new Calculator();
```

### Basic Arithmetic

#### add()
Adds two numbers.

```php
public function add(float $a, float $b): float

// Example
$result = $calculator->add(10, 5);  // Returns: 15
```

#### subtract()
Subtracts the second number from the first.

```php
public function subtract(float $a, float $b): float

// Example
$result = $calculator->subtract(10, 5);  // Returns: 5
```

#### multiply()
Multiplies two numbers.

```php
public function multiply(float $a, float $b): float

// Example
$result = $calculator->multiply(10, 5);  // Returns: 50
```

#### divide()
Divides the first number by the second.

```php
public function divide(float $a, float $b): float

// Example
$result = $calculator->divide(10, 5);  // Returns: 2

// Throws InvalidArgumentException if $b is 0
```

### Advanced Operations

#### power()
Raises a number to a power.

```php
public function power(float $base, float $exponent): float

// Example
$result = $calculator->power(2, 8);  // Returns: 256
$result = $calculator->power(4, 0.5);  // Returns: 2 (square root)
```

#### sqrt()
Calculates the square root.

```php
public function sqrt(float $number): float

// Example
$result = $calculator->sqrt(16);  // Returns: 4
$result = $calculator->sqrt(2);   // Returns: 1.414...

// Throws InvalidArgumentException if $number is negative
```

#### factorial()
Calculates the factorial of a number.

```php
public function factorial(int $n): int

// Example
$result = $calculator->factorial(5);  // Returns: 120
$result = $calculator->factorial(0);  // Returns: 1

// Throws InvalidArgumentException if $n is negative or > 170
```

#### modulo()
Calculates the remainder of division.

```php
public function modulo(float $a, float $b): float

// Example
$result = $calculator->modulo(10, 3);  // Returns: 1
$result = $calculator->modulo(7.5, 2);  // Returns: 1.5
```

#### percentage()
Calculates a percentage of a number.

```php
public function percentage(float $number, float $percent): float

// Example
$result = $calculator->percentage(200, 15);  // Returns: 30 (15% of 200)
$result = $calculator->percentage(100, 50);  // Returns: 50
```

#### abs()
Returns the absolute value.

```php
public function abs(float $number): float

// Example
$result = $calculator->abs(-10);  // Returns: 10
$result = $calculator->abs(5);    // Returns: 5
```

### Logarithmic Functions

#### log()
Calculates the base-10 logarithm.

```php
public function log(float $number): float

// Example
$result = $calculator->log(100);   // Returns: 2
$result = $calculator->log(1000);  // Returns: 3

// Throws InvalidArgumentException if $number <= 0
```

#### ln()
Calculates the natural logarithm (base e).

```php
public function ln(float $number): float

// Example
$result = $calculator->ln(M_E);    // Returns: 1
$result = $calculator->ln(2.718);  // Returns: ~1

// Throws InvalidArgumentException if $number <= 0
```

### Trigonometric Functions

All trigonometric functions use radians. Use `degToRad()` to convert from degrees.

#### sin()
Calculates the sine.

```php
public function sin(float $angle): float

// Example (using radians)
$result = $calculator->sin(M_PI / 2);  // Returns: 1 (90 degrees)
$result = $calculator->sin(0);         // Returns: 0

// Example (using degrees)
$degrees = 90;
$radians = $calculator->degToRad($degrees);
$result = $calculator->sin($radians);  // Returns: 1
```

#### cos()
Calculates the cosine.

```php
public function cos(float $angle): float

// Example
$result = $calculator->cos(0);      // Returns: 1
$result = $calculator->cos(M_PI);   // Returns: -1 (180 degrees)
```

#### tan()
Calculates the tangent.

```php
public function tan(float $angle): float

// Example
$result = $calculator->tan(M_PI / 4);  // Returns: 1 (45 degrees)
$result = $calculator->tan(0);         // Returns: 0
```

### Angle Conversion

#### degToRad()
Converts degrees to radians.

```php
public function degToRad(float $degrees): float

// Example
$result = $calculator->degToRad(180);  // Returns: 3.14159... (π)
$result = $calculator->degToRad(90);   // Returns: 1.5708... (π/2)
```

#### radToDeg()
Converts radians to degrees.

```php
public function radToDeg(float $radians): float

// Example
$result = $calculator->radToDeg(M_PI);      // Returns: 180
$result = $calculator->radToDeg(M_PI / 2);  // Returns: 90
```

### History Management

#### getHistory()
Returns the calculation history.

```php
public function getHistory(): array

// Example
$history = $calculator->getHistory();

// Returns array of operations:
// [
//     [
//         'operation' => 'add',
//         'operands' => [10, 5],
//         'result' => 15,
//         'timestamp' => '2026-02-04 12:00:00'
//     ],
//     ...
// ]
```

#### clearHistory()
Clears the calculation history.

```php
public function clearHistory(): void

// Example
$calculator->clearHistory();
```

## UnitConverter Class

### Namespace
```php
ScientificCalc\UnitConverter
```

### Constructor
```php
$converter = new UnitConverter($dataFile);

// Example
$converter = new UnitConverter(__DIR__ . '/data/conversions.json');
```

### Length Conversion

#### convertLength()
Converts between length units.

```php
public function convertLength(float $value, string $from, string $to): float

// Available units:
// millimeter, centimeter, decimeter, meter, decameter, hectometer, kilometer
// inch, foot, yard, mile, nautical_mile

// Examples
$feet = $converter->convertLength(1, 'meter', 'foot');         // 3.281 ft
$miles = $converter->convertLength(5, 'kilometer', 'mile');    // 3.107 mi
$cm = $converter->convertLength(10, 'inch', 'centimeter');     // 25.4 cm
```

### Time Conversion

#### convertTime()
Converts between time units.

```php
public function convertTime(float $value, string $from, string $to): float

// Available units:
// millisecond, second, minute, hour, day, week, month, year

// Examples
$seconds = $converter->convertTime(2, 'hour', 'second');      // 7200
$days = $converter->convertTime(48, 'hour', 'day');           // 2
$minutes = $converter->convertTime(1, 'day', 'minute');       // 1440
```

### Mass Conversion

#### convertMass()
Converts between mass/weight units.

```php
public function convertMass(float $value, string $from, string $to): float

// Available units:
// milligram, centigram, decigram, gram, decagram, hectogram, kilogram, metric_ton
// ounce, pound, stone, ton

// Examples
$pounds = $converter->convertMass(1, 'kilogram', 'pound');    // 2.205 lbs
$kg = $converter->convertMass(10, 'pound', 'kilogram');       // 4.536 kg
$ounces = $converter->convertMass(100, 'gram', 'ounce');      // 3.527 oz
```

### Temperature Conversion

#### convertTemperature()
Converts between temperature units.

```php
public function convertTemperature(float $value, string $from, string $to): float

// Available units:
// celsius, fahrenheit, kelvin, rankine

// Examples
$fahrenheit = $converter->convertTemperature(0, 'celsius', 'fahrenheit');    // 32°F
$celsius = $converter->convertTemperature(98.6, 'fahrenheit', 'celsius');    // 37°C
$kelvin = $converter->convertTemperature(0, 'celsius', 'kelvin');            // 273.15K
```

### Utility Methods

#### getAvailableUnits()
Returns available units for a conversion type.

```php
public function getAvailableUnits(string $type): array

// Example
$units = $converter->getAvailableUnits('length');
// Returns: ['millimeter', 'centimeter', 'decimeter', 'meter', ...]

$units = $converter->getAvailableUnits('temperature');
// Returns: ['celsius', 'fahrenheit', 'kelvin', 'rankine']
```

#### getConversionFactor()
Gets the conversion factor between two units.

```php
public function getConversionFactor(string $from, string $to, string $type): float

// Example
$factor = $converter->getConversionFactor('meter', 'foot', 'length');
// Returns: 3.28084 (1 meter = 3.28084 feet)
```

#### getHistory()
Returns the conversion history.

```php
public function getHistory(): array

// Returns array:
// [
//     [
//         'type' => 'length',
//         'value' => 10,
//         'from' => 'meter',
//         'to' => 'foot',
//         'result' => 32.8084,
//         'timestamp' => '2026-02-04 12:00:00'
//     ],
//     ...
// ]
```

#### saveHistory()
Saves the conversion history to JSON file.

```php
public function saveHistory(): bool

// Example
if ($converter->saveHistory()) {
    echo "History saved successfully!";
}
```

#### clearHistory()
Clears the conversion history.

```php
public function clearHistory(): void

// Example
$converter->clearHistory();
```

## Error Handling

All methods throw `InvalidArgumentException` for invalid inputs:

```php
try {
    $result = $calculator->divide(10, 0);
} catch (InvalidArgumentException $e) {
    echo "Error: " . $e->getMessage();
    // Output: "Division by zero is not allowed"
}

try {
    $result = $calculator->sqrt(-1);
} catch (InvalidArgumentException $e) {
    echo "Error: " . $e->getMessage();
    // Output: "Cannot calculate square root of negative number"
}

try {
    $result = $converter->convertLength(10, 'invalid', 'meter');
} catch (InvalidArgumentException $e) {
    echo "Error: " . $e->getMessage();
    // Output: "Unknown length unit: invalid"
}
```

## Complete Usage Example

```php
<?php

require_once 'src/Calculator.php';
require_once 'src/UnitConverter.php';

use ScientificCalc\Calculator;
use ScientificCalc\UnitConverter;

// Initialize
$calc = new Calculator();
$converter = new UnitConverter('data/conversions.json');

// Perform calculations
$sum = $calc->add(100, 50);
$power = $calc->power(2, 10);
$sine = $calc->sin($calc->degToRad(30));

// Convert units
$meters = $converter->convertLength(100, 'centimeter', 'meter');
$fahrenheit = $converter->convertTemperature(25, 'celsius', 'fahrenheit');
$pounds = $converter->convertMass(5, 'kilogram', 'pound');

// Display results
echo "Sum: $sum\n";
echo "2^10: $power\n";
echo "sin(30°): $sine\n";
echo "100cm = {$meters}m\n";
echo "25°C = {$fahrenheit}°F\n";
echo "5kg = {$pounds}lbs\n";

// Save conversion history
$converter->saveHistory();

// View history
$history = $converter->getHistory();
foreach ($history as $entry) {
    echo "{$entry['value']} {$entry['from']} = {$entry['result']} {$entry['to']}\n";
}
```

## Constants

Common mathematical constants available in PHP:

```php
M_PI        // π (3.14159...)
M_E         // e (2.71828...)
M_PI_2      // π/2
M_PI_4      // π/4
M_1_PI      // 1/π
M_2_PI      // 2/π
M_SQRT2     // √2
M_SQRT3     // √3
```

## Performance Considerations

- All conversions use efficient lookup tables
- History is automatically limited to prevent memory issues
- JSON file writes are optimized
- No external dependencies required

## Thread Safety

The classes are stateless except for history tracking. For multi-threaded environments:

1. Use separate instances per thread
2. Implement mutex locks for history writes
3. Consider using a database instead of JSON for production
