# Contributing to Scientific Calculator

Thank you for considering contributing to the Scientific Calculator project! We welcome contributions from everyone.

## How to Contribute

### Reporting Bugs

If you find a bug, please create an issue on GitHub with:

1. A clear, descriptive title
2. Steps to reproduce the bug
3. Expected behavior
4. Actual behavior
5. Screenshots (if applicable)
6. PHP version and environment details

### Suggesting Features

We welcome feature suggestions! Please create an issue with:

1. Clear description of the feature
2. Use case explaining why it's needed
3. Examples of how it would work
4. Any relevant mockups or designs

### Code Contributions

#### Getting Started

1. Fork the repository
2. Clone your fork:
```bash
git clone https://github.com/YOUR-USERNAME/scientific-calculator.git
cd scientific-calculator
```

3. Create a new branch:
```bash
git checkout -b feature/your-feature-name
```

#### Coding Standards

Follow these guidelines when writing code:

**PHP Standards:**
- Follow PSR-12 coding standards
- Use meaningful variable and function names
- Add PHPDoc comments for all public methods
- Keep functions focused and single-purpose
- Avoid deep nesting (max 3 levels)

**Example:**
```php
/**
 * Convert length between units
 *
 * @param float $value The value to convert
 * @param string $from Source unit
 * @param string $to Target unit
 * @return float Converted value
 * @throws InvalidArgumentException If units are invalid
 */
public function convertLength(float $value, string $from, string $to): float
{
    // Implementation
}
```

**Naming Conventions:**
- Classes: PascalCase (e.g., `Calculator`, `UnitConverter`)
- Methods: camelCase (e.g., `convertLength`, `getHistory`)
- Variables: camelCase (e.g., `$conversionFactor`, `$result`)
- Constants: UPPER_SNAKE_CASE (e.g., `MAX_HISTORY`, `BASE_UNITS`)

**File Structure:**
- One class per file
- Filename matches class name
- Use proper namespaces

#### Testing Your Changes

Before submitting:

1. Test all affected functionality
2. Run the test suite:
```bash
php tests/test.php
```

3. Test in multiple browsers if UI changes
4. Verify no PHP errors or warnings
5. Check that JSON storage still works

#### Making a Pull Request

1. Commit your changes:
```bash
git add .
git commit -m "Add feature: description of changes"
```

2. Push to your fork:
```bash
git push origin feature/your-feature-name
```

3. Create a Pull Request on GitHub with:
   - Clear title describing the change
   - Detailed description of what was changed and why
   - Reference to any related issues
   - Screenshots for UI changes
   - Test results

#### Pull Request Checklist

- [ ] Code follows PSR-12 standards
- [ ] All functions have PHPDoc comments
- [ ] Changes are tested and working
- [ ] No PHP errors or warnings
- [ ] Updated documentation if needed
- [ ] Added examples if adding new features
- [ ] Backwards compatible (or noted in PR)

## Development Setup

### Prerequisites
- PHP 7.4 or higher
- Git
- Text editor or IDE
- Web browser

### Local Development

1. Clone and navigate to the project:
```bash
git clone https://github.com/YOUR-USERNAME/scientific-calculator.git
cd scientific-calculator
```

2. Start development server:
```bash
cd public
php -S localhost:8000
```

3. Open http://localhost:8000 in your browser

### Project Structure

```
scientific-calculator/
├── src/              # Core PHP classes
├── public/           # Web interface
├── data/             # JSON storage
├── tests/            # Test files
├── config/           # Configuration
└── docs/             # Documentation
```

## Adding New Features

### Adding a Calculator Function

1. Add method to `src/Calculator.php`:
```php
public function newFunction(float $value): float
{
    $result = /* calculation */;
    $this->recordHistory('newFunction', [$value], $result);
    return $result;
}
```

2. Add UI button in `public/index.php`
3. Add test case in `tests/test.php`
4. Update documentation in `API.md`

### Adding a Unit Type

1. Add conversion factors to `src/UnitConverter.php`:
```php
private array $conversionFactors = [
    'newtype' => [
        'unit1' => 1,
        'unit2' => 1.5,
        // ...
    ],
];
```

2. Add conversion method:
```php
public function convertNewType(float $value, string $from, string $to): float
{
    $result = $this->convert($value, $from, $to, 'newtype');
    $this->recordConversion('newtype', $value, $from, $to, $result);
    return $result;
}
```

3. Update UI dropdowns
4. Add tests
5. Update documentation

## Code Review Process

All submissions require review. We aim to:

1. Review PRs within 3 days
2. Provide constructive feedback
3. Request changes if needed
4. Merge when approved

## Style Guide

### Commit Messages

Format:
```
Type: Brief description (50 chars or less)

Detailed explanation if needed. Wrap at 72 characters.
Reference issues like #123.
```

Types:
- `feat:` New feature
- `fix:` Bug fix
- `docs:` Documentation changes
- `style:` Code style changes (formatting)
- `refactor:` Code refactoring
- `test:` Adding tests
- `chore:` Maintenance tasks

Examples:
```
feat: Add exponential function to calculator

Implemented exp() function for calculating e^x.
Includes proper error handling and history tracking.
Closes #42
```

```
fix: Correct temperature conversion formula

Fixed Fahrenheit to Celsius conversion which was
returning incorrect values due to order of operations.
```

### Documentation

When adding features:

1. Update `README.md` if it affects usage
2. Update `API.md` with new methods
3. Add inline comments for complex logic
4. Include usage examples

## Questions?

- Create an issue for questions
- Tag issues with `question` label
- Be respectful and patient

## Recognition

Contributors will be:
- Listed in the README
- Mentioned in release notes
- Given credit in commit history

Thank you for contributing to make this project better! 🎉
