# Deployment Guide

## Quick Start Guide

### Method 1: PHP Built-in Server (Recommended for Development)

1. Navigate to the public directory:
```bash
cd public
```

2. Start the PHP development server:
```bash
php -S localhost:8000
```

3. Open your browser to:
```
http://localhost:8000
```

### Method 2: Apache/Nginx (Production)

#### Apache Setup

1. Install Apache and PHP:
```bash
sudo apt-get update
sudo apt-get install apache2 php libapache2-mod-php
```

2. Copy project to web directory:
```bash
sudo cp -r scientific-calculator /var/www/html/
```

3. Set permissions:
```bash
sudo chown -R www-data:www-data /var/www/html/scientific-calculator
sudo chmod -R 755 /var/www/html/scientific-calculator
sudo chmod 777 /var/www/html/scientific-calculator/data
```

4. Enable mod_rewrite:
```bash
sudo a2enmod rewrite
sudo systemctl restart apache2
```

5. Access via:
```
http://localhost/scientific-calculator/public/
```

#### Nginx Setup

1. Install Nginx and PHP-FPM:
```bash
sudo apt-get install nginx php-fpm
```

2. Create Nginx configuration file:
```bash
sudo nano /etc/nginx/sites-available/calculator
```

Add this configuration:
```nginx
server {
    listen 80;
    server_name calculator.local;
    root /var/www/html/scientific-calculator/public;
    index index.php;

    location / {
        try_files $uri $uri/ /index.php?$query_string;
    }

    location ~ \.php$ {
        fastcgi_pass unix:/var/run/php/php8.1-fpm.sock;
        fastcgi_index index.php;
        fastcgi_param SCRIPT_FILENAME $document_root$fastcgi_script_name;
        include fastcgi_params;
    }

    location ~ /\.ht {
        deny all;
    }
}
```

3. Enable the site:
```bash
sudo ln -s /etc/nginx/sites-available/calculator /etc/nginx/sites-enabled/
sudo nginx -t
sudo systemctl restart nginx
```

### Method 3: Docker (Cross-platform)

1. Create a Dockerfile in the project root:
```dockerfile
FROM php:8.1-apache

# Enable Apache modules
RUN a2enmod rewrite

# Copy application files
COPY . /var/www/html/scientific-calculator/

# Set permissions
RUN chown -R www-data:www-data /var/www/html/scientific-calculator
RUN chmod -R 755 /var/www/html/scientific-calculator
RUN chmod 777 /var/www/html/scientific-calculator/data

# Set working directory
WORKDIR /var/www/html/scientific-calculator/public

EXPOSE 80
```

2. Build and run:
```bash
docker build -t scientific-calculator .
docker run -p 8080:80 scientific-calculator
```

3. Access via:
```
http://localhost:8080/public/
```

## Troubleshooting

### Permission Issues

If you encounter permission errors with the data directory:

```bash
chmod 777 data/
chmod 666 data/conversions.json
```

### PHP Not Found

Make sure PHP is installed:
```bash
php --version
```

If not installed:
```bash
# Ubuntu/Debian
sudo apt-get install php php-cli

# macOS (using Homebrew)
brew install php

# Windows
Download from https://windows.php.net/download/
```

### File Not Found Errors

Make sure you're in the correct directory:
```bash
pwd  # Should show /path/to/scientific-calculator
ls   # Should show: src/, public/, data/, README.md
```

### JSON Storage Issues

If conversions aren't being saved:

1. Check file exists:
```bash
ls -la data/conversions.json
```

2. Check permissions:
```bash
chmod 666 data/conversions.json
```

3. Check directory permissions:
```bash
chmod 777 data/
```

## Production Checklist

Before deploying to production:

- [ ] Set proper file permissions (755 for directories, 644 for files)
- [ ] Make data directory writable (777 or 775)
- [ ] Review .htaccess security headers
- [ ] Enable HTTPS/SSL certificate
- [ ] Set error_reporting to 0 in PHP config
- [ ] Enable PHP opcode caching
- [ ] Configure proper session handling
- [ ] Set up regular backups for data/conversions.json
- [ ] Test all calculator functions
- [ ] Test all unit conversions
- [ ] Verify mobile responsiveness
- [ ] Check browser compatibility

## Performance Optimization

1. Enable PHP OPcache in php.ini:
```ini
opcache.enable=1
opcache.memory_consumption=128
opcache.max_accelerated_files=4000
opcache.revalidate_freq=60
```

2. Enable gzip compression in .htaccess (already included)

3. Set appropriate cache headers for static assets

## Security Best Practices

1. Keep PHP updated to the latest version
2. Disable directory listing (already configured)
3. Use HTTPS in production
4. Implement rate limiting for API endpoints
5. Validate and sanitize all user inputs (already implemented)
6. Regular security audits
7. Monitor server logs

## Monitoring

Monitor these files for issues:

- Apache error log: `/var/log/apache2/error.log`
- Nginx error log: `/var/log/nginx/error.log`
- PHP error log: Check php.ini for location
- Application data: `data/conversions.json`

## Backup Strategy

Regular backups of:

1. Application code (git repository)
2. Conversion history: `data/conversions.json`
3. Server configuration files

Recommended backup schedule:
- Daily: data/conversions.json
- Weekly: Full application backup
- Monthly: Server configuration

## Scaling Considerations

For high traffic scenarios:

1. Move JSON storage to a database (MySQL, PostgreSQL)
2. Implement caching with Redis or Memcached
3. Use load balancer for multiple application servers
4. CDN for static assets
5. Database connection pooling
6. API rate limiting
