# Scientific Calculator & Unit Converter

A comprehensive PHP-based scientific calculator with advanced unit conversion capabilities. Features a modern web interface with support for basic arithmetic, scientific functions, and conversions between various units of measurement.

## Features

### Scientific Calculator
- **Basic Operations**: Addition, subtraction, multiplication, division
- **Advanced Operations**: Power, square root, modulo, percentage
- **Scientific Functions**: 
  - Trigonometric functions (sin, cos, tan)
  - Logarithms (log₁₀, ln)
  - Factorial calculations
  - Support for both radians and degrees
- **Operation History**: Tracks recent calculations

### Unit Converter
- **Length**: millimeter, centimeter, decimeter, meter, decameter, hectometer, kilometer, inch, foot, yard, mile, nautical mile
- **Time**: millisecond, second, minute, hour, day, week, month, year
- **Mass**: milligram, centigram, decigram, gram, decagram, hectogram, kilogram, metric ton, ounce, pound, stone, ton
- **Temperature**: Celsius, Fahrenheit, Kelvin, Rankine
- **Conversion History**: Saves all conversions to JSON file

## Requirements

- PHP 7.4 or higher
- Web server (Apache, Nginx, or PHP built-in server)
- Write permissions for the `data` directory

## Installation

1. Clone the repository:
```bash
git clone <your-repo-url>
cd scientific-calculator
```

2. Set proper permissions:
```bash
chmod 755 data/
chmod 644 data/conversions.json
```

3. Start the application using one of these methods:

**Option A: PHP Built-in Server**
```bash
cd public
php -S localhost:8000
```

**Option B: Apache/Nginx**
Point your web server's document root to the `public` directory.

4. Open your browser and navigate to:
```
http://localhost:8000
```

## Directory Structure

```
scientific-calculator/
├── src/
│   ├── Calculator.php         # Core calculator class
│   └── UnitConverter.php      # Unit conversion engine
├── public/
│   └── index.php              # Main web interface
├── data/
│   └── conversions.json       # Conversion history storage
├── tests/                     # Unit tests (optional)
├── config/                    # Configuration files (optional)
└── README.md
```

## Usage Examples

### Calculator Operations

**Basic Arithmetic:**
```php
use ScientificCalc\Calculator;

$calc = new Calculator();

// Basic operations
$sum = $calc->add(10, 5);           // 15
$diff = $calc->subtract(10, 5);     // 5
$product = $calc->multiply(10, 5);  // 50
$quotient = $calc->divide(10, 5);   // 2

// Advanced operations
$power = $calc->power(2, 8);        // 256
$sqrt = $calc->sqrt(16);            // 4
$factorial = $calc->factorial(5);   // 120
```

**Scientific Functions:**
```php
// Trigonometry (radians)
$sine = $calc->sin(1.5708);         // ≈ 1 (90 degrees)
$cosine = $calc->cos(0);            // 1

// Convert degrees to radians
$radians = $calc->degToRad(90);     // ≈ 1.5708

// Logarithms
$log = $calc->log(100);             // 2
$ln = $calc->ln(2.71828);           // ≈ 1
```

### Unit Conversions

**Length Conversion:**
```php
use ScientificCalc\UnitConverter;

$converter = new UnitConverter();

// Convert meters to feet
$feet = $converter->convertLength(10, 'meter', 'foot');  // ≈ 32.81

// Convert kilometers to miles
$miles = $converter->convertLength(5, 'kilometer', 'mile');  // ≈ 3.11
```

**Temperature Conversion:**
```php
// Convert Celsius to Fahrenheit
$fahrenheit = $converter->convertTemperature(100, 'celsius', 'fahrenheit');  // 212

// Convert Kelvin to Celsius
$celsius = $converter->convertTemperature(273.15, 'kelvin', 'celsius');  // 0
```

**Mass Conversion:**
```php
// Convert kilograms to pounds
$pounds = $converter->convertMass(1, 'kilogram', 'pound');  // ≈ 2.205

// Convert grams to ounces
$ounces = $converter->convertMass(100, 'gram', 'ounce');  // ≈ 3.527
```

**Time Conversion:**
```php
// Convert hours to seconds
$seconds = $converter->convertTime(2, 'hour', 'second');  // 7200

// Convert days to minutes
$minutes = $converter->convertTime(1, 'day', 'minute');  // 1440
```

## API Reference

### Calculator Class

#### Basic Operations
- `add(float $a, float $b): float`
- `subtract(float $a, float $b): float`
- `multiply(float $a, float $b): float`
- `divide(float $a, float $b): float`

#### Advanced Operations
- `power(float $base, float $exponent): float`
- `sqrt(float $number): float`
- `modulo(float $a, float $b): float`
- `percentage(float $number, float $percent): float`
- `factorial(int $n): int`
- `abs(float $number): float`

#### Scientific Functions
- `sin(float $angle): float` - Sine (in radians)
- `cos(float $angle): float` - Cosine (in radians)
- `tan(float $angle): float` - Tangent (in radians)
- `log(float $number): float` - Logarithm base 10
- `ln(float $number): float` - Natural logarithm

#### Utility Functions
- `degToRad(float $degrees): float` - Convert degrees to radians
- `radToDeg(float $radians): float` - Convert radians to degrees
- `getHistory(): array` - Get calculation history
- `clearHistory(): void` - Clear calculation history

### UnitConverter Class

#### Conversion Methods
- `convertLength(float $value, string $from, string $to): float`
- `convertTime(float $value, string $from, string $to): float`
- `convertMass(float $value, string $from, string $to): float`
- `convertTemperature(float $value, string $from, string $to): float`

#### Utility Methods
- `getAvailableUnits(string $type): array` - Get list of available units
- `getConversionFactor(string $from, string $to, string $type): float`
- `getHistory(): array` - Get conversion history
- `clearHistory(): void` - Clear conversion history
- `saveHistory(): bool` - Save history to JSON file

## Error Handling

The application includes comprehensive error handling:

- **Division by zero**: Throws `InvalidArgumentException`
- **Negative square roots**: Throws `InvalidArgumentException`
- **Invalid logarithm inputs**: Throws `InvalidArgumentException`
- **Unknown units**: Throws `InvalidArgumentException`
- **Factorial overflow**: Limited to n ≤ 170

## Data Storage

Conversion history is stored in JSON format at `data/conversions.json`. The file structure:

```json
{
    "conversions": [
        {
            "type": "length",
            "value": 10,
            "from": "meter",
            "to": "foot",
            "result": 32.8084,
            "timestamp": "2026-02-04 12:00:00"
        }
    ],
    "last_updated": "2026-02-04 12:00:00"
}
```

## Contributing

1. Fork the repository
2. Create a feature branch (`git checkout -b feature/amazing-feature`)
3. Commit your changes (`git commit -m 'Add some amazing feature'`)
4. Push to the branch (`git push origin feature/amazing-feature`)
5. Open a Pull Request

## Testing

Run the built-in PHP server for testing:

```bash
cd public
php -S localhost:8000
```

Access the calculator at `http://localhost:8000`

## Security Considerations

- All user inputs are sanitized and validated
- Numeric inputs are cast to appropriate types
- HTML output is escaped to prevent XSS
- File operations use secure paths
- Session management for user-specific data

## Performance

- Efficient conversion algorithms using base unit approach
- Minimal database queries (uses JSON file storage)
- Optimized for quick calculations
- History limited to prevent memory issues

## Browser Compatibility

- Chrome (latest)
- Firefox (latest)
- Safari (latest)
- Edge (latest)
- Mobile browsers (iOS Safari, Chrome Mobile)

## License

MIT License - feel free to use this project for personal or commercial purposes.

## Author

Created with precision and attention to detail.

## Changelog

### Version 1.0.0
- Initial release
- Scientific calculator with basic and advanced operations
- Unit converter for length, time, mass, and temperature
- JSON-based history storage
- Responsive web interface

## Support

For issues, questions, or contributions, please open an issue on GitHub.
