<?php
require_once '../src/Calculator.php';
require_once '../src/UnitConverter.php';

use ScientificCalc\Calculator;
use ScientificCalc\UnitConverter;

session_start();

$calculator = new Calculator();
$converter = new UnitConverter(__DIR__ . '/../data/conversions.json');

$result = null;
$error = null;
$conversionResult = null;

// Handle calculator operations
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['calc_operation'])) {
    try {
        $operation = $_POST['calc_operation'];
        
        switch ($operation) {
            case 'add':
                $result = $calculator->add(floatval($_POST['num1']), floatval($_POST['num2']));
                break;
            case 'subtract':
                $result = $calculator->subtract(floatval($_POST['num1']), floatval($_POST['num2']));
                break;
            case 'multiply':
                $result = $calculator->multiply(floatval($_POST['num1']), floatval($_POST['num2']));
                break;
            case 'divide':
                $result = $calculator->divide(floatval($_POST['num1']), floatval($_POST['num2']));
                break;
            case 'power':
                $result = $calculator->power(floatval($_POST['num1']), floatval($_POST['num2']));
                break;
            case 'sqrt':
                $result = $calculator->sqrt(floatval($_POST['num1']));
                break;
            case 'log':
                $result = $calculator->log(floatval($_POST['num1']));
                break;
            case 'ln':
                $result = $calculator->ln(floatval($_POST['num1']));
                break;
            case 'sin':
                $angle = isset($_POST['angle_unit']) && $_POST['angle_unit'] === 'degrees' 
                    ? $calculator->degToRad(floatval($_POST['num1'])) 
                    : floatval($_POST['num1']);
                $result = $calculator->sin($angle);
                break;
            case 'cos':
                $angle = isset($_POST['angle_unit']) && $_POST['angle_unit'] === 'degrees' 
                    ? $calculator->degToRad(floatval($_POST['num1'])) 
                    : floatval($_POST['num1']);
                $result = $calculator->cos($angle);
                break;
            case 'tan':
                $angle = isset($_POST['angle_unit']) && $_POST['angle_unit'] === 'degrees' 
                    ? $calculator->degToRad(floatval($_POST['num1'])) 
                    : floatval($_POST['num1']);
                $result = $calculator->tan($angle);
                break;
            case 'factorial':
                $result = $calculator->factorial(intval($_POST['num1']));
                break;
            case 'percentage':
                $result = $calculator->percentage(floatval($_POST['num1']), floatval($_POST['num2']));
                break;
            case 'modulo':
                $result = $calculator->modulo(floatval($_POST['num1']), floatval($_POST['num2']));
                break;
        }
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}

// Handle unit conversions
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['convert'])) {
    try {
        $conversionType = $_POST['conversion_type'];
        $value = floatval($_POST['value']);
        $fromUnit = $_POST['from_unit'];
        $toUnit = $_POST['to_unit'];

        switch ($conversionType) {
            case 'length':
                $conversionResult = $converter->convertLength($value, $fromUnit, $toUnit);
                break;
            case 'time':
                $conversionResult = $converter->convertTime($value, $fromUnit, $toUnit);
                break;
            case 'mass':
                $conversionResult = $converter->convertMass($value, $fromUnit, $toUnit);
                break;
            case 'temperature':
                $conversionResult = $converter->convertTemperature($value, $fromUnit, $toUnit);
                break;
        }

        $converter->saveHistory();
    } catch (Exception $e) {
        $error = $e->getMessage();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Scientific Calculator & Unit Converter</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }

        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
            color: #333;
        }

        .container {
            max-width: 1200px;
            margin: 0 auto;
        }

        h1 {
            text-align: center;
            color: white;
            margin-bottom: 30px;
            font-size: 2.5em;
            text-shadow: 2px 2px 4px rgba(0,0,0,0.2);
        }

        .grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(500px, 1fr));
            gap: 20px;
            margin-bottom: 20px;
        }

        .card {
            background: white;
            border-radius: 15px;
            padding: 25px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.2);
        }

        h2 {
            color: #667eea;
            margin-bottom: 20px;
            border-bottom: 3px solid #667eea;
            padding-bottom: 10px;
        }

        .form-group {
            margin-bottom: 15px;
        }

        label {
            display: block;
            margin-bottom: 5px;
            font-weight: 600;
            color: #555;
        }

        input[type="number"],
        select {
            width: 100%;
            padding: 10px;
            border: 2px solid #ddd;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }

        input[type="number"]:focus,
        select:focus {
            outline: none;
            border-color: #667eea;
        }

        .button-grid {
            display: grid;
            grid-template-columns: repeat(4, 1fr);
            gap: 10px;
            margin-top: 15px;
        }

        button {
            padding: 15px;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
        }

        button:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(102, 126, 234, 0.4);
        }

        button:active {
            transform: translateY(0);
        }

        .result-box {
            background: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
            color: white;
            padding: 20px;
            border-radius: 8px;
            margin-top: 20px;
            font-size: 1.5em;
            text-align: center;
            font-weight: bold;
        }

        .error-box {
            background: #ff6b6b;
            color: white;
            padding: 15px;
            border-radius: 8px;
            margin-top: 20px;
        }

        .two-columns {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 15px;
        }

        .radio-group {
            display: flex;
            gap: 20px;
            margin-top: 10px;
        }

        .radio-group label {
            display: flex;
            align-items: center;
            gap: 5px;
            font-weight: normal;
        }

        .conversion-grid {
            display: grid;
            grid-template-columns: 1fr auto 1fr;
            gap: 10px;
            align-items: end;
        }

        .arrow {
            font-size: 24px;
            color: #667eea;
            align-self: center;
            padding-bottom: 30px;
        }

        @media (max-width: 768px) {
            .grid {
                grid-template-columns: 1fr;
            }
            
            .button-grid {
                grid-template-columns: repeat(3, 1fr);
            }
            
            h1 {
                font-size: 1.8em;
            }
        }

        .tab-container {
            margin-bottom: 20px;
        }

        .tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
        }

        .tab {
            padding: 10px 20px;
            background: #f0f0f0;
            border: none;
            border-radius: 8px 8px 0 0;
            cursor: pointer;
            font-weight: 600;
            transition: all 0.3s;
        }

        .tab.active {
            background: white;
            color: #667eea;
        }

        .tab-content {
            display: none;
        }

        .tab-content.active {
            display: block;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔬 Scientific Calculator & Unit Converter</h1>

        <?php if ($error): ?>
            <div class="error-box">
                ⚠️ Error: <?php echo htmlspecialchars($error); ?>
            </div>
        <?php endif; ?>

        <div class="grid">
            <!-- Calculator Section -->
            <div class="card">
                <h2>🧮 Scientific Calculator</h2>
                
                <div class="tab-container">
                    <div class="tabs">
                        <button class="tab active" onclick="switchTab('basic')">Basic</button>
                        <button class="tab" onclick="switchTab('scientific')">Scientific</button>
                    </div>

                    <!-- Basic Operations -->
                    <div id="basic" class="tab-content active">
                        <form method="POST">
                            <div class="two-columns">
                                <div class="form-group">
                                    <label>First Number:</label>
                                    <input type="number" name="num1" step="any" required>
                                </div>
                                <div class="form-group">
                                    <label>Second Number:</label>
                                    <input type="number" name="num2" step="any" required>
                                </div>
                            </div>
                            
                            <div class="button-grid">
                                <button type="submit" name="calc_operation" value="add">➕ Add</button>
                                <button type="submit" name="calc_operation" value="subtract">➖ Subtract</button>
                                <button type="submit" name="calc_operation" value="multiply">✖️ Multiply</button>
                                <button type="submit" name="calc_operation" value="divide">➗ Divide</button>
                                <button type="submit" name="calc_operation" value="power">^ Power</button>
                                <button type="submit" name="calc_operation" value="modulo">% Modulo</button>
                                <button type="submit" name="calc_operation" value="percentage">% Percentage</button>
                            </div>
                        </form>
                    </div>

                    <!-- Scientific Operations -->
                    <div id="scientific" class="tab-content">
                        <form method="POST">
                            <div class="form-group">
                                <label>Number:</label>
                                <input type="number" name="num1" step="any" required>
                            </div>

                            <div class="radio-group">
                                <label>
                                    <input type="radio" name="angle_unit" value="radians" checked>
                                    Radians
                                </label>
                                <label>
                                    <input type="radio" name="angle_unit" value="degrees">
                                    Degrees
                                </label>
                            </div>
                            
                            <div class="button-grid">
                                <button type="submit" name="calc_operation" value="sqrt">√ Square Root</button>
                                <button type="submit" name="calc_operation" value="log">log₁₀</button>
                                <button type="submit" name="calc_operation" value="ln">ln</button>
                                <button type="submit" name="calc_operation" value="factorial">n!</button>
                                <button type="submit" name="calc_operation" value="sin">sin</button>
                                <button type="submit" name="calc_operation" value="cos">cos</button>
                                <button type="submit" name="calc_operation" value="tan">tan</button>
                            </div>
                        </form>
                    </div>
                </div>

                <?php if ($result !== null): ?>
                    <div class="result-box">
                        Result: <?php echo number_format($result, 10); ?>
                    </div>
                <?php endif; ?>
            </div>

            <!-- Unit Converter Section -->
            <div class="card">
                <h2>🔄 Unit Converter</h2>
                
                <form method="POST">
                    <div class="form-group">
                        <label>Conversion Type:</label>
                        <select name="conversion_type" id="conversionType" onchange="updateUnits()" required>
                            <option value="">Select Type</option>
                            <option value="length">Length</option>
                            <option value="time">Time</option>
                            <option value="mass">Mass</option>
                            <option value="temperature">Temperature</option>
                        </select>
                    </div>

                    <div class="conversion-grid">
                        <div class="form-group">
                            <label>From:</label>
                            <input type="number" name="value" step="any" required placeholder="Enter value">
                            <select name="from_unit" id="fromUnit" required style="margin-top: 10px;">
                                <option value="">Select unit</option>
                            </select>
                        </div>

                        <div class="arrow">→</div>

                        <div class="form-group">
                            <label>To:</label>
                            <select name="to_unit" id="toUnit" required>
                                <option value="">Select unit</option>
                            </select>
                        </div>
                    </div>

                    <button type="submit" name="convert" style="width: 100%; margin-top: 15px;">
                        Convert
                    </button>
                </form>

                <?php if ($conversionResult !== null): ?>
                    <div class="result-box">
                        <?php 
                        echo number_format($_POST['value'], 6) . ' ' . htmlspecialchars($_POST['from_unit']) 
                             . ' = ' . number_format($conversionResult, 6) . ' ' . htmlspecialchars($_POST['to_unit']); 
                        ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script>
        const units = {
            length: ['millimeter', 'centimeter', 'decimeter', 'meter', 'decameter', 'hectometer', 'kilometer', 'inch', 'foot', 'yard', 'mile', 'nautical_mile'],
            time: ['millisecond', 'second', 'minute', 'hour', 'day', 'week', 'month', 'year'],
            mass: ['milligram', 'centigram', 'decigram', 'gram', 'decagram', 'hectogram', 'kilogram', 'metric_ton', 'ounce', 'pound', 'stone', 'ton'],
            temperature: ['celsius', 'fahrenheit', 'kelvin', 'rankine']
        };

        function updateUnits() {
            const type = document.getElementById('conversionType').value;
            const fromUnit = document.getElementById('fromUnit');
            const toUnit = document.getElementById('toUnit');

            fromUnit.innerHTML = '<option value="">Select unit</option>';
            toUnit.innerHTML = '<option value="">Select unit</option>';

            if (type && units[type]) {
                units[type].forEach(unit => {
                    fromUnit.innerHTML += `<option value="${unit}">${unit.replace('_', ' ')}</option>`;
                    toUnit.innerHTML += `<option value="${unit}">${unit.replace('_', ' ')}</option>`;
                });
            }
        }

        function switchTab(tabName) {
            const tabs = document.querySelectorAll('.tab');
            const contents = document.querySelectorAll('.tab-content');

            tabs.forEach(tab => tab.classList.remove('active'));
            contents.forEach(content => content.classList.remove('active'));

            event.target.classList.add('active');
            document.getElementById(tabName).classList.add('active');
        }
    </script>
</body>
</html>
