<?php

namespace ScientificCalc;

/**
 * Scientific Calculator Class
 * Provides basic arithmetic and advanced scientific operations
 */
class Calculator
{
    private array $history = [];
    private const MAX_HISTORY = 50;

    /**
     * Add two numbers
     */
    public function add(float $a, float $b): float
    {
        $result = $a + $b;
        $this->recordHistory('add', [$a, $b], $result);
        return $result;
    }

    /**
     * Subtract two numbers
     */
    public function subtract(float $a, float $b): float
    {
        $result = $a - $b;
        $this->recordHistory('subtract', [$a, $b], $result);
        return $result;
    }

    /**
     * Multiply two numbers
     */
    public function multiply(float $a, float $b): float
    {
        $result = $a * $b;
        $this->recordHistory('multiply', [$a, $b], $result);
        return $result;
    }

    /**
     * Divide two numbers
     */
    public function divide(float $a, float $b): float
    {
        if ($b == 0) {
            throw new \InvalidArgumentException("Division by zero is not allowed");
        }
        $result = $a / $b;
        $this->recordHistory('divide', [$a, $b], $result);
        return $result;
    }

    /**
     * Calculate power (a^b)
     */
    public function power(float $base, float $exponent): float
    {
        $result = pow($base, $exponent);
        $this->recordHistory('power', [$base, $exponent], $result);
        return $result;
    }

    /**
     * Calculate square root
     */
    public function sqrt(float $number): float
    {
        if ($number < 0) {
            throw new \InvalidArgumentException("Cannot calculate square root of negative number");
        }
        $result = sqrt($number);
        $this->recordHistory('sqrt', [$number], $result);
        return $result;
    }

    /**
     * Calculate logarithm (base 10)
     */
    public function log(float $number): float
    {
        if ($number <= 0) {
            throw new \InvalidArgumentException("Logarithm undefined for non-positive numbers");
        }
        $result = log10($number);
        $this->recordHistory('log', [$number], $result);
        return $result;
    }

    /**
     * Calculate natural logarithm
     */
    public function ln(float $number): float
    {
        if ($number <= 0) {
            throw new \InvalidArgumentException("Natural logarithm undefined for non-positive numbers");
        }
        $result = log($number);
        $this->recordHistory('ln', [$number], $result);
        return $result;
    }

    /**
     * Calculate sine (radians)
     */
    public function sin(float $angle): float
    {
        $result = sin($angle);
        $this->recordHistory('sin', [$angle], $result);
        return $result;
    }

    /**
     * Calculate cosine (radians)
     */
    public function cos(float $angle): float
    {
        $result = cos($angle);
        $this->recordHistory('cos', [$angle], $result);
        return $result;
    }

    /**
     * Calculate tangent (radians)
     */
    public function tan(float $angle): float
    {
        $result = tan($angle);
        $this->recordHistory('tan', [$angle], $result);
        return $result;
    }

    /**
     * Convert degrees to radians
     */
    public function degToRad(float $degrees): float
    {
        return deg2rad($degrees);
    }

    /**
     * Convert radians to degrees
     */
    public function radToDeg(float $radians): float
    {
        return rad2deg($radians);
    }

    /**
     * Calculate factorial
     */
    public function factorial(int $n): int
    {
        if ($n < 0) {
            throw new \InvalidArgumentException("Factorial not defined for negative numbers");
        }
        if ($n > 170) {
            throw new \InvalidArgumentException("Factorial too large to calculate");
        }
        
        $result = 1;
        for ($i = 2; $i <= $n; $i++) {
            $result *= $i;
        }
        
        $this->recordHistory('factorial', [$n], $result);
        return $result;
    }

    /**
     * Calculate absolute value
     */
    public function abs(float $number): float
    {
        return abs($number);
    }

    /**
     * Calculate modulo
     */
    public function modulo(float $a, float $b): float
    {
        if ($b == 0) {
            throw new \InvalidArgumentException("Modulo by zero is not allowed");
        }
        return fmod($a, $b);
    }

    /**
     * Calculate percentage
     */
    public function percentage(float $number, float $percent): float
    {
        return ($number * $percent) / 100;
    }

    /**
     * Record operation in history
     */
    private function recordHistory(string $operation, array $operands, $result): void
    {
        $this->history[] = [
            'operation' => $operation,
            'operands' => $operands,
            'result' => $result,
            'timestamp' => date('Y-m-d H:i:s')
        ];

        // Keep only last MAX_HISTORY entries
        if (count($this->history) > self::MAX_HISTORY) {
            array_shift($this->history);
        }
    }

    /**
     * Get calculation history
     */
    public function getHistory(): array
    {
        return $this->history;
    }

    /**
     * Clear history
     */
    public function clearHistory(): void
    {
        $this->history = [];
    }
}
