<?php

namespace ScientificCalc;

/**
 * Unit Converter Class
 * Handles conversions for length, time, temperature, and mass
 */
class UnitConverter
{
    private array $conversionHistory = [];
    private string $dataFile;

    // Base units for conversion (everything converts to/from these)
    private const BASE_UNITS = [
        'length' => 'meter',
        'time' => 'second',
        'temperature' => 'celsius',
        'mass' => 'kilogram'
    ];

    // Conversion factors to base units
    private array $conversionFactors = [
        'length' => [
            'millimeter' => 0.001,
            'centimeter' => 0.01,
            'decimeter' => 0.1,
            'meter' => 1,
            'decameter' => 10,
            'hectometer' => 100,
            'kilometer' => 1000,
            'inch' => 0.0254,
            'foot' => 0.3048,
            'yard' => 0.9144,
            'mile' => 1609.344,
            'nautical_mile' => 1852,
        ],
        'time' => [
            'millisecond' => 0.001,
            'second' => 1,
            'minute' => 60,
            'hour' => 3600,
            'day' => 86400,
            'week' => 604800,
            'month' => 2628000, // average month (30.4167 days)
            'year' => 31536000, // 365 days
        ],
        'mass' => [
            'milligram' => 0.000001,
            'centigram' => 0.00001,
            'decigram' => 0.0001,
            'gram' => 0.001,
            'decagram' => 0.01,
            'hectogram' => 0.1,
            'kilogram' => 1,
            'metric_ton' => 1000,
            'ounce' => 0.0283495,
            'pound' => 0.453592,
            'stone' => 6.35029,
            'ton' => 907.185, // US ton
        ],
    ];

    public function __construct(string $dataFile = '../data/conversions.json')
    {
        $this->dataFile = $dataFile;
        $this->loadHistory();
    }

    /**
     * Convert length units
     */
    public function convertLength(float $value, string $from, string $to): float
    {
        $result = $this->convert($value, $from, $to, 'length');
        $this->recordConversion('length', $value, $from, $to, $result);
        return $result;
    }

    /**
     * Convert time units
     */
    public function convertTime(float $value, string $from, string $to): float
    {
        $result = $this->convert($value, $from, $to, 'time');
        $this->recordConversion('time', $value, $from, $to, $result);
        return $result;
    }

    /**
     * Convert mass units
     */
    public function convertMass(float $value, string $from, string $to): float
    {
        $result = $this->convert($value, $from, $to, 'mass');
        $this->recordConversion('mass', $value, $from, $to, $result);
        return $result;
    }

    /**
     * Convert temperature units
     */
    public function convertTemperature(float $value, string $from, string $to): float
    {
        $from = strtolower($from);
        $to = strtolower($to);

        // Temperature conversion requires special handling
        $celsius = $this->toCelsius($value, $from);
        $result = $this->fromCelsius($celsius, $to);

        $this->recordConversion('temperature', $value, $from, $to, $result);
        return $result;
    }

    /**
     * Generic conversion method
     */
    private function convert(float $value, string $from, string $to, string $type): float
    {
        $from = strtolower($from);
        $to = strtolower($to);

        if (!isset($this->conversionFactors[$type][$from])) {
            throw new \InvalidArgumentException("Unknown {$type} unit: {$from}");
        }

        if (!isset($this->conversionFactors[$type][$to])) {
            throw new \InvalidArgumentException("Unknown {$type} unit: {$to}");
        }

        // Convert to base unit, then to target unit
        $baseValue = $value * $this->conversionFactors[$type][$from];
        $result = $baseValue / $this->conversionFactors[$type][$to];

        return $result;
    }

    /**
     * Convert any temperature to Celsius
     */
    private function toCelsius(float $value, string $from): float
    {
        switch ($from) {
            case 'celsius':
                return $value;
            case 'fahrenheit':
                return ($value - 32) * 5 / 9;
            case 'kelvin':
                return $value - 273.15;
            case 'rankine':
                return ($value - 491.67) * 5 / 9;
            default:
                throw new \InvalidArgumentException("Unknown temperature unit: {$from}");
        }
    }

    /**
     * Convert Celsius to any temperature
     */
    private function fromCelsius(float $celsius, string $to): float
    {
        switch ($to) {
            case 'celsius':
                return $celsius;
            case 'fahrenheit':
                return ($celsius * 9 / 5) + 32;
            case 'kelvin':
                return $celsius + 273.15;
            case 'rankine':
                return ($celsius + 273.15) * 9 / 5;
            default:
                throw new \InvalidArgumentException("Unknown temperature unit: {$to}");
        }
    }

    /**
     * Get available units for a type
     */
    public function getAvailableUnits(string $type): array
    {
        $type = strtolower($type);
        
        if ($type === 'temperature') {
            return ['celsius', 'fahrenheit', 'kelvin', 'rankine'];
        }

        if (!isset($this->conversionFactors[$type])) {
            throw new \InvalidArgumentException("Unknown conversion type: {$type}");
        }

        return array_keys($this->conversionFactors[$type]);
    }

    /**
     * Record conversion in history
     */
    private function recordConversion(string $type, float $value, string $from, string $to, float $result): void
    {
        $this->conversionHistory[] = [
            'type' => $type,
            'value' => $value,
            'from' => $from,
            'to' => $to,
            'result' => $result,
            'timestamp' => date('Y-m-d H:i:s')
        ];

        // Keep only last 100 conversions
        if (count($this->conversionHistory) > 100) {
            array_shift($this->conversionHistory);
        }
    }

    /**
     * Get conversion history
     */
    public function getHistory(): array
    {
        return $this->conversionHistory;
    }

    /**
     * Clear history
     */
    public function clearHistory(): void
    {
        $this->conversionHistory = [];
    }

    /**
     * Save history to JSON file
     */
    public function saveHistory(): bool
    {
        $directory = dirname($this->dataFile);
        if (!is_dir($directory)) {
            mkdir($directory, 0755, true);
        }

        $data = json_encode([
            'conversions' => $this->conversionHistory,
            'last_updated' => date('Y-m-d H:i:s')
        ], JSON_PRETTY_PRINT);

        return file_put_contents($this->dataFile, $data) !== false;
    }

    /**
     * Load history from JSON file
     */
    private function loadHistory(): void
    {
        if (file_exists($this->dataFile)) {
            $data = json_decode(file_get_contents($this->dataFile), true);
            if (isset($data['conversions']) && is_array($data['conversions'])) {
                $this->conversionHistory = $data['conversions'];
            }
        }
    }

    /**
     * Get conversion factor between two units
     */
    public function getConversionFactor(string $from, string $to, string $type): float
    {
        if ($type === 'temperature') {
            // For temperature, we need to convert 1 unit
            return $this->convertTemperature(1, $from, $to);
        }

        return $this->convert(1, $from, $to, $type);
    }
}
