<?php

/**
 * Example Test File for Calculator Class
 * 
 * This is a basic test file to demonstrate functionality.
 * For production, consider using PHPUnit or another testing framework.
 */

require_once __DIR__ . '/../src/Calculator.php';
require_once __DIR__ . '/../src/UnitConverter.php';

use ScientificCalc\Calculator;
use ScientificCalc\UnitConverter;

echo "=== Scientific Calculator Tests ===\n\n";

$calc = new Calculator();

// Test basic operations
echo "Basic Operations:\n";
echo "10 + 5 = " . $calc->add(10, 5) . " (Expected: 15)\n";
echo "10 - 5 = " . $calc->subtract(10, 5) . " (Expected: 5)\n";
echo "10 * 5 = " . $calc->multiply(10, 5) . " (Expected: 50)\n";
echo "10 / 5 = " . $calc->divide(10, 5) . " (Expected: 2)\n";
echo "\n";

// Test advanced operations
echo "Advanced Operations:\n";
echo "2 ^ 8 = " . $calc->power(2, 8) . " (Expected: 256)\n";
echo "√16 = " . $calc->sqrt(16) . " (Expected: 4)\n";
echo "5! = " . $calc->factorial(5) . " (Expected: 120)\n";
echo "10 % 3 = " . $calc->modulo(10, 3) . " (Expected: 1)\n";
echo "\n";

// Test scientific functions
echo "Scientific Functions:\n";
echo "sin(π/2) = " . $calc->sin(M_PI / 2) . " (Expected: ~1)\n";
echo "cos(0) = " . $calc->cos(0) . " (Expected: 1)\n";
echo "log₁₀(100) = " . $calc->log(100) . " (Expected: 2)\n";
echo "ln(e) = " . $calc->ln(M_E) . " (Expected: ~1)\n";
echo "\n";

// Test angle conversion
echo "Angle Conversion:\n";
echo "90° to radians = " . $calc->degToRad(90) . " (Expected: ~1.5708)\n";
echo "π radians to degrees = " . $calc->radToDeg(M_PI) . " (Expected: 180)\n";
echo "\n";

// Test error handling
echo "Error Handling Tests:\n";
try {
    $calc->divide(10, 0);
    echo "FAIL: Division by zero should throw exception\n";
} catch (Exception $e) {
    echo "PASS: Division by zero caught: " . $e->getMessage() . "\n";
}

try {
    $calc->sqrt(-1);
    echo "FAIL: Negative square root should throw exception\n";
} catch (Exception $e) {
    echo "PASS: Negative square root caught: " . $e->getMessage() . "\n";
}

echo "\n=== Unit Converter Tests ===\n\n";

$converter = new UnitConverter(__DIR__ . '/../data/conversions.json');

// Test length conversions
echo "Length Conversions:\n";
echo "1 meter to feet = " . $converter->convertLength(1, 'meter', 'foot') . " (Expected: ~3.281)\n";
echo "1 kilometer to miles = " . $converter->convertLength(1, 'kilometer', 'mile') . " (Expected: ~0.621)\n";
echo "10 centimeters to inches = " . $converter->convertLength(10, 'centimeter', 'inch') . " (Expected: ~3.937)\n";
echo "\n";

// Test time conversions
echo "Time Conversions:\n";
echo "1 hour to seconds = " . $converter->convertTime(1, 'hour', 'second') . " (Expected: 3600)\n";
echo "1 day to minutes = " . $converter->convertTime(1, 'day', 'minute') . " (Expected: 1440)\n";
echo "60 seconds to minutes = " . $converter->convertTime(60, 'second', 'minute') . " (Expected: 1)\n";
echo "\n";

// Test mass conversions
echo "Mass Conversions:\n";
echo "1 kilogram to pounds = " . $converter->convertMass(1, 'kilogram', 'pound') . " (Expected: ~2.205)\n";
echo "100 grams to ounces = " . $converter->convertMass(100, 'gram', 'ounce') . " (Expected: ~3.527)\n";
echo "1 pound to grams = " . $converter->convertMass(1, 'pound', 'gram') . " (Expected: ~453.592)\n";
echo "\n";

// Test temperature conversions
echo "Temperature Conversions:\n";
echo "0°C to Fahrenheit = " . $converter->convertTemperature(0, 'celsius', 'fahrenheit') . " (Expected: 32)\n";
echo "100°C to Fahrenheit = " . $converter->convertTemperature(100, 'celsius', 'fahrenheit') . " (Expected: 212)\n";
echo "273.15K to Celsius = " . $converter->convertTemperature(273.15, 'kelvin', 'celsius') . " (Expected: 0)\n";
echo "32°F to Celsius = " . $converter->convertTemperature(32, 'fahrenheit', 'celsius') . " (Expected: 0)\n";
echo "\n";

// Test available units
echo "Available Units:\n";
echo "Length units: " . implode(', ', $converter->getAvailableUnits('length')) . "\n";
echo "Time units: " . implode(', ', $converter->getAvailableUnits('time')) . "\n";
echo "Mass units: " . implode(', ', $converter->getAvailableUnits('mass')) . "\n";
echo "Temperature units: " . implode(', ', $converter->getAvailableUnits('temperature')) . "\n";
echo "\n";

// Save history
if ($converter->saveHistory()) {
    echo "Conversion history saved successfully!\n";
} else {
    echo "Failed to save conversion history.\n";
}

echo "\n=== All Tests Completed ===\n";
